clear all
close all
clc

fclose('all');
format long

% ========================================================== %
%         Viscoelastic Parameter Extraction Script
%
%  Submitted in conjunction with the following manuscript:
%  "Extracting Viscoelastic Material Parameters using an
%  Atomic Force Microscope and Static Force Spectroscopy"
% 
%            See readme file for usage details.
% ========================================================== %

%% Load SFS Files and Parse
addpath Igor2Matlab/

pathname = uigetdir('Select an AFM Data Directory');

% FilesCheck stores the number of ibw files in the directory.
% If you are not using the .ibw format, change "ibw" in the script below to
% the appropriate filetype. NOTE: this will likely break other parts of the
% script, especially when using ibwread().
FilesCheck = dir([pathname '\*.ibw']);
FilesCheck = FilesCheck(~ismember({FilesCheck.name},{'.','..'}));

% Begin looping through the files and parsing the information stored in the
% filenames.
if length(FilesCheck) > 1
    % The case where there is more than one experimental dataset
    FilesTemp = FilesCheck(1).name;
    FilesTemp = strsplit(FilesTemp, {'_' '.'},'CollapseDelimiters',true);
    
    % If using a new type of file identification system, ensure that the
    % FIRST term of the underscore-separated filename is a unique
    % identifier that can be implemented like the types below. If you are
    % unsure of which filing system to use, try the FD format. This
    % involves naming your files using the format: FD_(Point #)_(Run
    % #)_XXX-xx.ibw. Note that XXX-xx is the approach velocity, with the 
    % decimal replaced by a hyphen (-).
    
    switch FilesTemp{1} 
        case 'FD'
            Files=dir([pathname '\*.ibw']);
            for k=1:length(Files)
                FileNames = Files(k).name;
                FileInfo = strsplit(FileNames, {'_' '.'},'CollapseDelimiters',true);

                v_approach(k) = str2num(strrep(FileInfo{4},'-','.'))*1E-9;  % Approach Velocity, nm/s
                point_number(k) = str2num(strrep(FileInfo{2},'-','.'));     % Position Number
                run_number(k) = str2num(strrep(FileInfo{3},'-','.'))+1;     % Run Number
            end
        case 'ELopez'
            Files=dir([pathname '\*.ibw']);
            for k=1:length(Files)
                FileNames = Files(k).name;
                FileInfo = strsplit(FileNames, {'_' '.'},'CollapseDelimiters',true);

                v_approach(k) = str2num(strrep(FileInfo{4},'-','.'))*1E-9;  % Approach Velocity, nm/s
                point_number(k) = str2num(strrep(FileInfo{2},'-','.'));     % Position Number
                run_number(k) = str2num(strrep(FileInfo{3},'-','.'))+1;     % Run Number
            end
        case 'ELopezSim'
            Files=dir([pathname '\*.txt']);
            for k=1:length(Files)
                FileNames = Files(k).name;
                FileInfo = strsplit(FileNames, {'_' '.'},'CollapseDelimiters',true);

                v_approach(k) = str2num(strrep(FileInfo{4},'-','.'))*1E-9;  % Approach Velocity, nm/s
                point_number(k) = str2num(strrep(FileInfo{2},'-','.'));     % Position Number
                run_number(k) = str2num(strrep(FileInfo{3},'-','.'))+1;     % Run Number
            end
    end
    
elseif length(FilesCheck) == 1
    
    FilesTemp = FilesCheck.name;
    FilesTemp = strsplit(FilesTemp, {'_' '.'},'CollapseDelimiters',true);

    switch FilesTemp{1}
        case 'FD'
            Files=dir([pathname '\*.ibw']);
            FileNames = Files.name;
            FileInfo = strsplit(FileNames, {'_' '.'},'CollapseDelimiters',true);

            v_approach = str2num(strrep(FileInfo{4},'-','.'))*1E-9;         % Approach Velocity, nm/s
            point_number = str2num(strrep(FileInfo{2},'-','.'));            % Position Number
            run_number = str2num(strrep(FileInfo{3},'-','.'))+1;            % Run Number

        case 'ELopez'
            Files=dir([pathname '\*.ibw']);
            FileNames = Files.name;
            FileInfo = strsplit(FileNames, {'_' '.'},'CollapseDelimiters',true);

            v_approach = str2num(strrep(FileInfo{4},'-','.'))*1E-9;         % Approach Velocity, nm/s
            point_number = str2num(strrep(FileInfo{2},'-','.'));            % Position Number
            run_number = str2num(strrep(FileInfo{3},'-','.'))+1;            % Run Number

        case 'ELopezSim'
            Files=dir([pathname '\*.txt']);
            FileNames = Files.name;
            FileInfo = strsplit(FileNames, {'_' '.'},'CollapseDelimiters',true);

            v_approach = str2num(strrep(FileInfo{4},'-','.'))*1E-9;         % Approach Velocity, nm/s
            point_number = str2num(strrep(FileInfo{2},'-','.'));            % Position Number
            run_number = str2num(strrep(FileInfo{3},'-','.'))+1;            % Run Number
    end
else
   
    fprintf('No files found in the directory. Exiting script.\n\n');
    return
    
end

% Create the datastruct utilized for saving all data streams.
dataStruct = struct;

% Parse each file found in the directory chosen earlier.
for k=1:length(Files)
    
    % Choose the k-th file to analyze.
    FileNames = Files(k).name;
    FileInfo = strsplit(FileNames, {'_' '.'},'CollapseDelimiters',true);
    
    % Get the k-th file's name information and then use the ibwread()
    % function to extract the wavenote information.
    switch FileInfo{1}
        case 'FD'
            RawData = IBWread([Files(k).folder '\' Files(k).name]);
            [headerValue,null] = strsplit(RawData.WaveNotes,'\r',...
            'DelimiterType','RegularExpression');
            
            % Get the filename information
            v_approach_temp = str2num(strrep(FileInfo{4},'-','.'))*1E-9;    % Approach Velocity, m/s
            point_number_temp = str2num(strrep(FileInfo{2},'-','.'));       % Position Number
            run_number_temp = str2num(strrep(FileInfo{3},'-','.'))+1;       % Run Number
            
            % Save the datastreams for this file to the structure 
            dataStruct(k).z = RawData.y(:,1);
            dataStruct(k).d = RawData.y(:,2);
            
            % Find the Spring Constant
            varIndex = find(contains(headerValue,'SpringConstant'),1);
            temp = headerValue{varIndex};
            temp(isspace(temp)) = [];
            temp = split(temp,':');
            k_cantilever(k) = str2num(temp{2}); % Value in N/m

            % Find the Deflection InvOLS
            varIndex = find(contains(headerValue,'InvOLS'));
            temp = headerValue{varIndex};
            temp(isspace(temp)) = [];
            temp = split(temp,':');
            defl_InVOLS(k) = str2num(temp{2}); % Value in nm/V
            
            % Create the raw time array
            varIndex = find(contains(headerValue,'NumPtsPerSec'));
            temp = headerValue{varIndex};
            temp(isspace(temp)) = [];
            temp = split(temp,':');
            dataStruct(k).dt = 1/str2num(temp{2}); % Value in s
            dataStruct(k).n_sam = RawData.Nsam;
            dataStruct(k).t = dataStruct(k).dt.*((1:size(dataStruct(k).d))-1)';
    
        case 'ELopez'
            RawData = IBWread([Files(k).folder '\' Files(k).name]);
            [headerValue,null] = strsplit(RawData.WaveNotes,'\r',...
            'DelimiterType','RegularExpression');

            v_approach_temp = str2num(strrep(FileInfo{4},'-','.'))*1E-9;    % Approach Velocity, m/s
            point_number_temp = str2num(strrep(FileInfo{2},'-','.'));       % Position Number
            run_number_temp = str2num(strrep(FileInfo{3},'-','.'))+1;       % Run Number
            
            % Save the data streams to the structure
            dataStruct(k).z = RawData.y(:,5);
            dataStruct(k).d = RawData.y(:,2);
            
            % Find the Spring Constant
            varIndex = find(contains(headerValue,'SpringConstant'),1);
            temp = headerValue{varIndex};
            temp(isspace(temp)) = [];
            temp = split(temp,':');
            k_cantilever(k) = str2num(temp{2}); % Value in N/m

            % Find the Deflection InvOLS
            varIndex = find(contains(headerValue,'InvOLS'));
            temp = headerValue{varIndex};
            temp(isspace(temp)) = [];
            temp = split(temp,':');
            defl_InVOLS(k) = str2num(temp{2}); % Value in nm/V
            
            % Create time array
            varIndex = find(contains(headerValue,'NumPtsPerSec'));
            temp = headerValue{varIndex};
            temp(isspace(temp)) = [];
            temp = split(temp,':');
            dataStruct(k).dt = 1/str2num(temp{2}); % Value in s
            dataStruct(k).n_sam = RawData.Nsam;
            dataStruct(k).t = dataStruct(k).dt.*((1:size(dataStruct(k).d))-1)';
                
        case 'ELopezSim'
            
            % To utilize the files located in Lopez et al.'s github
            % repository, a different reading algorithm is necessary.
            if length(Files) > 1
                fileID = fopen(fullfile([Files(k).folder '\' Files(k).name]), 'rt');
                RawData = textscan(fileID,'%f%f%f%f%f','Delimiter',' ','HeaderLines',1);
                fclose(fileID);
            else
                fileID = fopen([Files.folder '\' Files.name], 'rt');
                RawData = textscan(fileID,'%f%f%f%f%f','Delimiter',' ','HeaderLines',1);
                fclose(fileID);
            end
            
            % Get the filename information
            v_approach_temp = str2num(strrep(FileInfo{4},'-','.'))*1E-9;    % Approach Velocity, m/s
            point_number_temp = str2num(strrep(FileInfo{2},'-','.'));       % Position Number
            run_number_temp = str2num(strrep(FileInfo{3},'-','.'))+1;       % Run Number
            
            % Save the data streams to the structure
            dataStruct(k).z = RawData{2}.*1E-9;
            dataStruct(k).d = RawData{5}.*1E-9;
            
            % Programmed directly from the notes contained in Lopez et
            % al.'s github repository.
            k_cantilever(k) = 0.1;                                          % Value in N/m

            % Insert Fake InVOLS (Not needed for this analysis)
            defl_InVOLS(k) = 1;                                             % Value in nm/V
                
            % Create time array from information on Github repository
            dataStruct(k).dt = 1E-4;                                        % Value in s
            dataStruct(k).n_sam = size(RawData{1},2);
            dataStruct(k).t = RawData{1};
            
    end
    
    % Pre-Processing
    % Find the approach portion of the data
    [z_max, z_max_ind] = max(dataStruct(k).z);
    dataStruct(k).z_approach = dataStruct(k).z(1:z_max_ind);
    dataStruct(k).d_approach = dataStruct(k).d(1:z_max_ind);
    dataStruct(k).t_approach = dataStruct(k).t(1:z_max_ind);
    
    % Calculate Deflection Offset
    [null, d_min_ind] = min(dataStruct(k).d_approach);
    indScale = 0.9;
    d_0_mean = mean(dataStruct(k).d_approach(1:round(d_min_ind*indScale)));
    dataStruct(k).d_corrected = dataStruct(k).d_approach - d_0_mean;
    
    % Filter and Shift z_sensor data
    % Create the butterworth
    fs = 0.005*(1/dataStruct(k).dt);
    N = 2; % First order
    cutoff_Hz = (1/max(dataStruct(k).t));                                   % Should be 3dB down at the cutoff
    [b,a] = butter(N,cutoff_Hz/(fs/2),'low');                               % This makes a lowpass filter
    d_smooth = (filter(b,a,dataStruct(k).d_corrected));                     % Next, apply the filter
    [~, dSmoothMin] = min(d_smooth);
    
    if abs(d_smooth(dSmoothMin) - min(dataStruct(k).d_corrected))/abs(min(dataStruct(k).d_corrected)) > 0.01
        [~, dSmoothMin] = min(dataStruct(k).d_corrected);
    end
    
    % Calculate z_corrected (Equation 27)
    dataStruct(k).z_corrected = dataStruct(k).z_approach - ...
        dataStruct(k).z_approach(dSmoothMin) + ...
        dataStruct(k).d_corrected(dSmoothMin);
    
    % Save the index where the minimum appears to occur and the smooth data
    dataStruct(k).dSmoothMin = dSmoothMin;
    dataStruct(k).d_smooth = d_smooth;
    
    % Save the minimum deflection point offsets (d0, z0)
    dataStruct(k).d0 = dataStruct(k).d_corrected(dSmoothMin);
    dataStruct(k).z0 = dataStruct(k).z_corrected(dSmoothMin);
    
    % Calculate Force and Indentation (Equations 28 & 29)
    dataStruct(k).F = dataStruct(k).d_corrected .* k_cantilever(k);
    dataStruct(k).F_smooth = d_smooth .* k_cantilever(k);
    dataStruct(k).h = (dataStruct(k).z(1:z_max_ind) - dataStruct(k).z0)...
        - (dataStruct(k).d(1:z_max_ind) - dataStruct(k).d0);
    
    % Get Repulsive Portion of the Data
    % First, get the number of points between the minimum deflection and
    % maximum z-sensor readings.
    n_offset = length(dataStruct(k).d_corrected(dSmoothMin:z_max_ind));
    
    % Use the timestep from this particular file to create the repulsive
    % time vector, t_rep
    dt = dataStruct(k).dt;    
    t_rep = linspace(0,(n_offset-1)*dt,n_offset)';
    
    % Extract the repulsive z-sensor, deflection, and smoothed deflection
    % (for use, if desired).
    z_rep = dataStruct(k).z_corrected(dSmoothMin:end);
    d_rep = dataStruct(k).d_corrected(dSmoothMin:end);
    d_rep_smooth = dataStruct(k).d_smooth(dSmoothMin:end);
    
    % Since the deflection has already been shifted, we can use the point 
    % of zero deflection to pick exactly where force application begins.
    % The notation "tip_rep" is a legacy choice from Lopez et al.'s script,
    % which indicates tip_rep as repulsive tip position.
    tip_rep = d_rep;
    
    % Get the point where the deflected tip position is positive.
    tip_rep_pos = find(tip_rep > 0, 1);
    
    % Save the repulsive time, z-sensor, and deflections that are
    % normalized by their initial magnitudes by subtracting their first
    % values. This allows all of the curves to start at the origin (0,0).
    dataStruct(k).t_r = t_rep(tip_rep > 0) - t_rep(tip_rep_pos);
    dataStruct(k).z_r = z_rep(tip_rep > 0) - z_rep(tip_rep_pos);
    dataStruct(k).d_r = d_rep(tip_rep > 0) - d_rep(tip_rep_pos);
    dataStruct(k).d_r_smooth = d_rep_smooth(tip_rep > 0) - d_rep_smooth(tip_rep_pos);
    
    % Calculate Force and Indentation during Repulsive Portion of data, and
    % save them to the structure.
    dataStruct(k).F_r = dataStruct(k).d_r .* k_cantilever(k);               % Calculate Force
    dataStruct(k).h_r = dataStruct(k).z_r - dataStruct(k).d_r;              % Calculate Indentation

    dataStruct(k).F_r(dataStruct(k).h_r < 0) = 0;                           % Fix negative indentation forces
    dataStruct(k).F_r_smooth = dataStruct(k).d_r_smooth .* k_cantilever(k); % Calculate Smooth Force
    
    dataStruct(k).h_r(dataStruct(k).h_r < 0) = 0;                           % Fix negative indentation
    dataStruct(k).h_r_smooth = dataStruct(k).z_r -dataStruct(k).d_r_smooth; % Calculate Smooth Indentation
    
    % Store the point of maximum z-sensor in the structure
    dataStruct(k).z_max_ind = z_max_ind;

    % Change to Logarithmic Scaling to use Enrique et al.'s separate 
    % function method.
    tr = dataStruct(k).dt;
    st = dataStruct(k).t(z_max_ind);
    
    % Initialize logarithmic data arrays
    F_log = [];
    t_log = [];
    
    % Get logarithmic data streams from log_scale() function.
    F_log = log_scale(dataStruct(k).F_r,dataStruct(k).t_r,tr,st);
    t_log = log_scale(dataStruct(k).t_r,dataStruct(k).t_r,tr,st);
        
    % Save the Log-Form of the tip force (F) and time array (t_r)
    dataStruct(k).F_r_log = F_log;
    dataStruct(k).t_r_log = t_log;
        
    % Memory management and ensuring unique values are used between files.
    clearvars d_0_mean d_smooth d_min_ind dSmoothMin F_log FileInfo...
        FileNames headerValue Index null n_offset RawData temp...
        v_approach_temp varIndex z_max z_max_ind t_log...
        tip_log titleString
    
end

% Memory management
clearvars a b cutoff_Hz fs k N point_number_temp run_number_temp

% Determine how many approach velocities we are considering
v_unique = sort(unique(v_approach));

% Calculating the averaged datasets
if length(Files) > 1
    FileNames = Files(1).name;
    FileInfo = strsplit(FileNames, {'_' '.'},'CollapseDelimiters',true);
    
    % Find the number of files
    N = length(Files);
    
    % If you arrive at this point and any of the following variables are
    % throwing an error (i.e. unable to perform assignment becuase the left
    % and right sides have a different number of elements), check if the
    % variable stored in dataStruct is empty. It's possible one of the
    % files needs to be ignored manually (put in a subfolder and left
    % unused because it's essentially useless, containing no repulsive
    % information or at least none clasically identifiable). Look in
    % "Files" to determine what the name is of each row in dataStruct with
    % an empty .t_r variable and move it to the ignored folder.
    
    for k=1:length(Files)
        minVal(k) = min(dataStruct(k).z_corrected);
        maxVal(k) = max(dataStruct(k).z_corrected);
        dtVal(k) = dataStruct(k).dt;
        maxTimeVal(k) = max(dataStruct(k).t_approach);
    end
    
    [startNum,~] = max(minVal);
    [endNum,~] = min(maxVal);
    [minHighTime,~] = min(maxTimeVal);
    
    filePoints = ceil(minHighTime/median(dtVal));
        
    xi = linspace(startNum, endNum, filePoints);                            % Create Vector Of Common Z-Values
    yi = [];
    ti = [];
    
    if length(v_unique) == 1
        % Case when only one velocity is present
        
        for k=1:length(Files)
            % Use z_corrected and d_corrected from this file to "look-up"
            % the general z-position array xi.
            [tempz, indz] = unique(dataStruct(k).z_corrected);
            yi(:,k) = interp1(tempz, dataStruct(k).d_corrected(indz),...
                xi(:), 'linear', 0);                                        % Interploate Or Extrapolate To New x Values
            
            % Create an associated time array for averaging
            ti(:,k) = spline(tempz,((1:length(tempz)).*dataStruct(k).dt),xi(:));
            
            % Normalize the current column by it's starting point for
            % averaging
            ti(:,k) = ti(:,k)-ti(1,k);
        end
        
        % Save the averaged/spliced datastreams for analysis
        dataStruct(length(Files)+1).z_average = xi';
        dataStruct(length(Files)+1).d_average = mean(yi,2);
        dataStruct(length(Files)+1).t_average = mean(ti,2);
        dataStruct(length(Files)+1).dt = mean(diff(dataStruct(length(Files)+1).t_average));
        
    else
        
        % Loop through unique velocities
        for j=1:length(v_unique)
            
            % Loop through all files
            for k=1:length(Files)
                
                % Check if this file is relevant for this specific
                % averaging operation
                if v_approach(k) == v_unique(j) && length(dataStruct(k).t_r) > 5
                    [tempz, indz] = unique(dataStruct(k).z_corrected);
                    yi = horzcat(yi, interp1(tempz, dataStruct(k).d_corrected(indz),...
                        xi(:), 'linear', 0)); % Interploate Or Extrapolate To New x Values
                    dtList = horzcat(dataStruct(k).dt);
                    
                    % Create an associated time array for averaging and
                    % then normalize it by the starting point.
                    titemp = spline(tempz,...
                        ((1:length(tempz)).*dataStruct(k).dt),xi(:));
                    ti = horzcat(ti, titemp-titemp(1));
                    
                else
                    continue;
                end
                
            end
            
            % Average this velocity's curves and save them to a unique row
            % for that velocity.
            dataStruct(length(Files)+j).z_average = xi';
            dataStruct(length(Files)+j).d_average = mean(yi,2);
            dataStruct(length(Files)+j).t_average = mean(ti,2);
            dataStruct(length(Files)+j).dt = mean(diff(dataStruct(length(Files)+j).t_average));
            
        end
    end
    
else
    
    % For the case where only one file is present, it IS the averaged
    % dataset. This is mostly so settings later (average or individual
    % analysis) don't break.
    dataStruct(length(Files)+1).z_average = dataStruct(1).z_corrected;
    dataStruct(length(Files)+1).d_average = dataStruct(1).d_corrected;
    dataStruct(length(Files)+1).t_average = dataStruct(1).t_approach;
    dataStruct(length(Files)+1).dt = mean(diff(dataStruct(length(Files)+1).t_average));
    
end

% Pre-Processing for Averaged Data of all load levels.
for i = 1:length(v_unique)
    % Find the approach portion of the data. This is only really necessary
    % if the average data happens to go past the limit of all the datasets
    % for some reason.
    [z_max, z_max_ind] = max(dataStruct(length(Files)+i).z_average);
    dataStruct(length(Files)+i).z_approach = dataStruct(length(Files)+i).z_average(1:z_max_ind);
    dataStruct(length(Files)+i).d_approach = dataStruct(length(Files)+i).d_average(1:z_max_ind);
    
    % Find the minimum deflection point
    [~, dSmoothMin] = min(dataStruct(length(Files)+i).d_approach);

    % Get Repulsive Portion of the Data
    dzs = mean(gradient(dataStruct(length(Files)+i).z_average));
    n_offset = length(dataStruct(length(Files)+i).d_average(dSmoothMin:z_max_ind));
    
    % Save the repulsive time, z-sensor, and deflections.
    t_rep = dataStruct(length(Files)+i).t_average;
    z_rep = dataStruct(length(Files)+i).z_average(dSmoothMin:z_max_ind);
    d_rep = dataStruct(length(Files)+i).d_average(dSmoothMin:z_max_ind);
    
    % Use tip position to find where force application begins.
    tip_rep = d_rep;
    tip_rep_pos = find(tip_rep > 0, 1);
    
    % Find the repulsive portion (force application) region
    dataStruct(length(Files)+i).t_r = t_rep(tip_rep > 0) - t_rep(tip_rep_pos);
    dataStruct(length(Files)+i).z_r = z_rep(tip_rep > 0) - z_rep(tip_rep_pos);
    dataStruct(length(Files)+i).d_r = d_rep(tip_rep > 0) - d_rep(tip_rep_pos);

    % Calculate Force and Indentation during Repulsive Portion
    dataStruct(length(Files)+i).F_r = dataStruct(length(Files)+i).d_r .* mean(k_cantilever);
    dataStruct(length(Files)+i).h_r = dataStruct(length(Files)+i).z_r - dataStruct(length(Files)+i).d_r; % Calculate Indentation
    
    dataStruct(length(Files)+i).F_r(dataStruct(length(Files)+i).h_r < 0) = 0;
    dataStruct(length(Files)+i).h_r(dataStruct(length(Files)+i).h_r < 0) = 0; % Fix negative indentations
    
    % Save the maximum z-sensor and minimum deflection indices
    dataStruct(length(Files)+i).z_max_ind = z_max_ind;
    dataStruct(length(Files)+i).dSmoothMin = dSmoothMin;

    % Change to Logarithmic Scaling to use Enrique et al.'s Fit Method
    tr = dataStruct(length(Files)+i).dt;
    st = dataStruct(length(Files)+i).t_r(end);
    
    % Initialize logarithmic arrays
    F_log = [];
    t_log = [];
    
    % Get logarithmic scaled data from log_scale()
    F_log = log_scale(dataStruct(length(Files)+i).F_r,dataStruct(length(Files)+i).t_r,tr,st);
    t_log = log_scale(dataStruct(length(Files)+i).t_r,dataStruct(length(Files)+i).t_r,tr,st);

    % Save the Log-Form of the tip force (F) and time array (t_r)
    dataStruct(length(Files)+i).F_r_log = F_log;
    dataStruct(length(Files)+i).t_r_log = t_log;
    
end

clearvars d_r_average t_r_average z_r_average N z_max_ind_temp temp d_temp t_temp z_temp xi yi 

%% Visualize the Data Together
close all

% These loops are used to visualize all of the datasets compared to the
% averaged dataset. This allows the user to confirm that the data is
% actually being modeled appropriately.

% First, plot the full datasets versus the average approximation
for i = 1:length(v_unique)
    
    legendString = {};
    figure('Position',  [10, 50, 1000, 1000])    
    hold on
    grid on
    
    titleString = sprintf('Deflection vs. Z-Sensor\nv_{app} = %4.2f [nm/s]',1E9*v_unique(i));
    
    % Plot each dataset and save the corresponding series label for the
    % legend later on.
    if length(Files) > 1
        for ii = 1:length(Files)
            % Importantly: skip this data series if it isn't part of the
            % velocity group that you're plotting.
            if v_approach(ii) == v_unique(i)
                legendString = vertcat(legendString, {sprintf('Point %d, Run %d',point_number(ii),run_number(ii))});
                plot(dataStruct(ii).z_corrected,dataStruct(ii).d_corrected)
            else
                continue;
            end
        end
    else 
        % If it's only one file, it's necessarily only one velocity and
        % dataset to plot.
        legendString = vertcat(legendString, {sprintf('Point %d, Run %d',point_number,run_number)});
        plot(dataStruct(i).z_corrected,dataStruct(i).d_corrected)
    end
    
    % Plot average data
    legendString = vertcat(legendString, {sprintf('Averaged Curve')});
    plot(dataStruct(length(Files)+i).z_average,dataStruct(length(Files)+i).d_average,'k--','LineWidth',5)
    
    % Manage the figure settings
    set(findall(gcf,'-property','FontSize'),'FontSize',28)
    set(gca,'TickLength',[0.02 0.02],'LineWidth',3)
    xlabel('Z-sensor [m]', 'FontSize', 32)
    ylabel('Deflection [m]', 'FontSize', 32)
%     title(titleString, 'FontSize', 32)
%     legend(legendString,'Location','eastoutside','Orientation','Vertical', 'FontSize', 10)
    hold off
    
    % Save the figure to the TempFigs folder.
    % NOTE: This script will break if there isn't a TempFigs folder and you
    % attempt to uncomment and use these lines. They are left commented out
    % until the user wants to save the figures.
%     saveas(gcf, sprintf('TempFigs/AFM-SFS-CorrectedDataset-Velocity_%.f_nm-s',v_unique(i)/1E-9), 'jpg')
%     savefig(gcf, sprintf('TempFigs/AFM-SFS-CorrectedDataset-Velocity_%.f_nm-s.fig',v_unique(i)/1E-9),'compact')

end

clearvars legendString titleString

% Second, plot ONLY the repulsive portion of the curves for comparison.
for i = 1:length(v_unique)
    
    legendString = {};
    figure('Position',  [10, 50, 1000, 1000])
    hold on
    grid on
    
    titleString = sprintf('Repulsive Deflection vs. Repulsive Z-Sensor\nv_{app} = %4.2f [nm/s]',1E9*v_unique(i));
    
    % Plot the data
    if length(Files) > 1
        for ii = 1:length(Files)
            if v_approach(ii) == v_unique(i) && length(dataStruct(ii).t_r) > 5
                legendString = vertcat(legendString, {sprintf('Point %d, Run %d',point_number(ii),run_number(ii))});
                plot(dataStruct(ii).z_r,dataStruct(ii).d_r,'LineWidth',5)
            else
                continue;
            end
        end
    else 
        legendString = vertcat(legendString, {sprintf('Point %d, Run %d',point_number,run_number)});
        plot(dataStruct(i).z_r,dataStruct(i).d_r,'LineWidth',5)
    end
    
    % Plot the average
    legendString = vertcat(legendString, {sprintf('Averaged Curve')});
    plot(dataStruct(length(Files)+i).z_r,dataStruct(length(Files)+i).d_r,'k--','LineWidth',5)
    
    % Change the figure settings
    set(findall(gcf,'-property','FontSize'),'FontSize',28)
    set(gca,'TickLength',[0.02 0.02],'LineWidth',3)
    xlabel('Repulsive Z-sensor [m]', 'FontSize', 32)
    ylabel('Repulsive Deflection [m]', 'FontSize', 32)
%     title(titleString, 'FontSize', 20)
%     legend(legendString,'Location','eastoutside','Orientation','Vertical', 'FontSize', 10)
    hold off
    
    % Save the figure to the TempFigs folder.
    % NOTE: This script will break if there isn't a TempFigs folder and you
    % attempt to uncomment and use these lines. They are left commented out
    % until the user wants to save the figures.
%     saveas(gcf, sprintf('TempFigs/AFM-SFS-RepulsiveDataset-Velocity_%.f_nm-s',v_unique(i)/1E-9), 'jpg')
%     savefig(gcf, sprintf('TempFigs/AFM-SFS-RepulsiveDataset-Velocity_%.f_nm-s.fig',v_unique(i)/1E-9),'compact')

end

% Plot the indentation (h) datasets and the average
for i = 1:length(v_unique)
    
    legendString = {};
    figure('Position',  [10, 50, 1000, 1000])
    hold on
    grid on
    
    titleString = sprintf('Indentation vs. Time\nv_{app} = %4.2f [nm/s]',1E9*v_unique(i));
    
    % Plot the datasets
    if length(Files) > 1
        for ii = 1:length(Files)
            if v_approach(ii) == v_unique(i) && length(dataStruct(ii).t_r) > 5
                legendString = vertcat(legendString, {sprintf('Point %d, Run %d',point_number(ii),run_number(ii))});
                plot(dataStruct(ii).t_r,dataStruct(ii).h_r, 'LineWidth', 5);
            else
                continue;
            end
        end
    else 
        legendString = vertcat(legendString, {sprintf('Point %d, Run %d',point_number,run_number)});
        plot(dataStruct(i).t_r,dataStruct(i).h_r)
    end
    
    % Plot the average
    legendString = vertcat(legendString, {sprintf('Averaged Curve')});
    plot(dataStruct(length(Files)+i).t_r,dataStruct(length(Files)+i).h_r,'k--','LineWidth',5)
    
    % Change the figure settings
    set(findall(gcf,'-property','FontSize'),'FontSize',28)
    set(gca,'TickLength',[0.02 0.02],'LineWidth',3)
    xlabel('Repulsive Time [s]', 'FontSize', 32)
    ylabel('Repulsive Indentation [m]', 'FontSize', 32)
%     title(titleString, 'FontSize', 32)
%     legend(legendString,'Location','eastoutside','Orientation','Vertical', 'FontSize', 10)
    hold off
    
    % Save the figure to the TempFigs folder.
    % NOTE: This script will break if there isn't a TempFigs folder and you
    % attempt to uncomment and use these lines. They are left commented out
    % until the user wants to save the figures.
%     saveas(gcf, sprintf('TempFigs/AFM-SFS-IndentationDataset-Velocity_%.f_nm-s',v_unique(i)/1E-9), 'jpg')
%     savefig(gcf, sprintf('TempFigs/AFM-SFS-IndentationDataset-Velocity_%.f_nm-s.fig',v_unique(i)/1E-9),'compact')

end

%% Acquire the fitting analysis settings
% If you want to close the plots, uncomment the following line:
% close all
clc

r_tip = input('\nPlease enter the radius of the indenter in nanometers: ');
while r_tip < 0 || r_tip > 10000
    r_tip = input('\nYour number is invalid, please enter an appropriate radius in nanometers: ');
end
r_tip = r_tip * 1E-9;

n_terms_tot = input('\nPlease enter the number of voigt terms you would like to use (1-5): ');
while n_terms_tot > 5 || n_terms_tot < 1
    n_terms_tot = input('\nYour number is outside the range (1-5), please enter a valid number: ');
end

multiSim = input('\nWould you like to run this file for each Voigt term number below your selection?\n(i.e. for 5-term, this will run for 1, 2, 3, and 4 terms, too) (y/n): ','s');
while multiSim ~= 'n' && multiSim ~= 'y'
    multiSim = input('\nYour answer is not acceptable. Please enter (y) or (n): ','s');
end

if multiSim == 'y'
    nSims = n_terms_tot;
    voigtArray = 1:n_terms_tot;
else
    nSims = 1;
    voigtArray = n_terms_tot;
end

skipNums = input('\nWould you like to skip any numbers? If so, please enter them in array form here, or use empty brackets ([]): ');

avAnalysis = input('\nAre you using individual(i) or averaged analysis(a)?: ','s');
while avAnalysis ~= 'i' && avAnalysis ~= 'a'
    avAnalysis = input('\Please enter (i) for individual or (a) for averaged: ','s');
end

scaling = input('\nWhat scaling factor would you like to use on the fitting tolerances?: ');
while scaling < 0 || scaling > 1 || ischar(scaling)
    scaling = input('\nYour number is outside the range (0,1) or is not a number, please enter a valid factor: ');
end

saveMats = input('\nWould you like to save .mat files? (y/n): ','s');
while saveMats ~= 'n' && saveMats ~= 'y'
    saveMats = input('\nYour answer is not acceptable. Please enter (y) or (n): ','s');
end

%% Loop through simulations

% Get out of any open files
fclose('all');

% This loop will run through each simulation, changing the number of voigt
% terms used in each case according the settings above.
for iVoigt = 1:nSims
    
    % Notify the user of the current run information.
    fprintf('\nRunning Simulation for %d Retardance Terms\n\n',voigtArray(iVoigt));
    n_terms = voigtArray(iVoigt);
    
    % Skip simulation if that number of terms is being disregarded
    if ismember(n_terms,skipNums)
       fprintf('\nSkipped %d Retardance Terms per user request.\n\n',voigtArray(iVoigt));
       continue; 
    end
    
    % Data Fitting with Lopez et al.
    Figure(1) = figure('position',[50 120 1400 550]);

    % Open Parallel Pool of MATLAB Workers
    if isempty(gcp('nocreate'))
        parpool
    end
    
    % These bounds and initial guesses are used for all simulations if the
    % user wishes to change these to something more appropriate for their
    % model, then the order of variables is:
    % [J_g J_1 tau_1 J_2 tau_2 J_3 tau_3 J_4 tau_4 J_5 tau_5]
    lb_all = [0 0 1e-5 0 1e-4 0 1e-3 0 1e-2 0 1e-1];
    ub_all = [1 1 1e-3 1 1e-2 1 1e-1 1 1e0 1 1e1];
    initParams_all = [1.0e-9 1.0e-10 1.0e-4 1.0e-10 1.0e-3 1.0e-10 1.0e-2 1.0e-10 1.0e-1 1.0e-10 1.0e0];
    
    % Create the text file where the simulation parameters are saved for 
    % consideration later on.
    fid = fopen([pathname sprintf('\\FitParams-nVoigt_%d.txt',voigtArray(iVoigt))],'w');
    fprintf(fid,'Fitting Parameters, %s\r\n=================\r\n', date);
    fprintf(fid,'\r\nTip Radius: %4.3g\r\nVoigt Terms: %d\r\nScaling Factor: %4.3g',r_tip,n_terms,scaling);
    
    % Switch the cases for either averaged dataset or individual file
    % dataset analysis.
    switch avAnalysis
        case 'i'
            fprintf(fid,'Individual File Analysis\r\n\r\n');
            
            % Loop through all files
            for ii = 1:length(Files)
                fprintf(fid,'File No. %d:\r\n============',ii);
                
                % The linear fit takes place first. This is in line with
                % the analysis performed by Lopez et al. for the case where
                % linear Force vs. Time curves is acceptable.
                x_lin = dataStruct(ii).t_r_log;                             % X data for linear fit
                y_lin = dataStruct(ii).F_r_log;                             % Y data for linear fit

                init_params = polyfit(x_lin, y_lin, 1);                     % Initial Linear Fit Parameters (guess)

                % Linear Fit Model Definition
                F_linFit = @(c,input) c .* input;

                lb = 0;                                                     % Zero Newtons/s is the minimum force rate possible
                ub = 1e4;                                                   % 1e4 Newton/s is the upper bound
                
                % Settings for the fitting operation
                options = optimoptions('lsqcurvefit','Algorithm','trust-region-reflective',...
                    'MaxFunctionEvaluations', 1e3,...
                    'MaxIterations', 1e3,...
                    'FiniteDifferenceType','central',...
                    'FunctionTolerance', 1e-35,...
                    'OptimalityTolerance',1e-35,...
                    'StepTolerance', 1e-35,...
                    'Display', 'none');

                n_samples = 1e3;
                beta_dist = zeros(length(lb),n_samples);
                beta0_dist = zeros(size(beta_dist));
                resnorm_dist = zeros(1,n_samples);

                progressString = sprintf('Linear Force Assumption, Curve #%d\nInvestigating Fdot\nParallel Search Running...',ii);
                hbar = parfor_progressbar(n_samples,progressString);
                warning('off');

                % Parallel Loop for linear case grid search
                for i = 1:n_samples
                    try
                        beta0 = init_params(1);
                        [betatemp,resnorm,residual,exitflag,output,lambda,jacobian] = ...
                            lsqcurvefit(F_linFit,beta0,x_lin,y_lin,lb,ub,options);

                        beta_dist(:,i) = betatemp;
                        beta0_dist(:,i) = beta0;
                        resnorm_dist(i) = sum(abs(residual));
                    catch
                        beta_dist(:,i) = NaN;
                        beta0_dist(:,i) = NaN;
                        resnorm_dist(i) = NaN;
                    end
                        hbar.iterate(1) % Increase progressbar by 1 iteration
                end

                close(hbar)
                warning('on');

                [minRes,idx] = min(resnorm_dist);
                fprintf(fid,'\r\n\nThe Lowest Normal Residual (resnorm) for the Linear fit was %3.4e, using:\r\n',minRes);
                fprintf(fid,'Fdot = %3.6g\r\n',beta_dist(1,idx));
                dataStruct(ii).Fdot = beta_dist(1,idx);

                % Calculate the Linear Fluidity (chi)
                % According to Eq. 14 from Lopez et al., This is the 
                % relation between chi and tip location when force is 
                % assumed to be linear.

                dt = dataStruct(ii).dt;
                st = dataStruct(ii).t_r(end);

                dataStruct(ii).chi_linear = ((16.0*sqrt(r_tip))/3)...
                    .* (dataStruct(ii).h_r.^1.5) ./ dataStruct(ii).Fdot;
                dataStruct(ii).chi_linear_log = log_scale(dataStruct(ii).chi_linear,...
                    dataStruct(ii).t_r, dt, st);

                % Function Shapes for Multiple Voigt with Linear Load Assumption
                switch n_terms
                    case 1
                        F_linear_fit = @(c,inputs) log_scale(( c(2).*c(3).*(exp(-inputs(:,1)./c(3))-1) )...
                            + (c(1)+c(2))*inputs(:,1),...
                            dataStruct(ii).t_r,dt,st);

                    case 2
                        F_linear_fit = @(c,inputs) log_scale(( c(2).*c(3).*(exp(-inputs(:,1)./c(3))-1)...
                            + c(4).*c(5).*(exp(-inputs(:,1)./c(5))-1) )...
                            + (c(1)+c(2)+c(4))*inputs(:,1),...
                            dataStruct(ii).t_r,dt,st);

                    case 3
                        F_linear_fit = @(c,inputs) log_scale(( c(2).*c(3).*(exp(-inputs(:,1)./c(3))-1)...
                            + c(4).*c(5).*(exp(-inputs(:,1)./c(5))-1)...
                            + c(6).*c(7).*(exp(-inputs(:,1)./c(7))-1) )...
                            + (c(1)+c(2)+c(4)+c(6))*inputs(:,1),...
                            dataStruct(ii).t_r,dt,st);

                    case 4
                        F_linear_fit = @(c,inputs) log_scale(( c(2).*c(3).*(exp(-inputs(:,1)./c(3))-1)...
                            + c(4).*c(5).*(exp(-inputs(:,1)./c(5))-1)...
                            + c(6).*c(7).*(exp(-inputs(:,1)./c(7))-1)...
                            + c(8).*c(9).*(exp(-inputs(:,1)./c(9))-1) )...
                            + (c(1)+c(2)+c(4)+c(6)+c(8))*inputs(:,1),...
                            dataStruct(ii).t_r,dt,st);

                    case 5
                        F_linear_fit = @(c,inputs) log_scale(( c(2).*c(3).*(exp(-inputs(:,1)./c(3))-1)...
                            + c(4).*c(5).*(exp(-inputs(:,1)./c(5))-1)...
                            + c(6).*c(7).*(exp(-inputs(:,1)./c(7))-1)...
                            + c(8).*c(9).*(exp(-inputs(:,1)./c(9))-1)...
                            + c(10).*c(11).*(exp(-inputs(:,1)./c(11))-1) )...
                            + (c(1)+c(2)+c(4)+c(6)+c(8)+c(10))*inputs(:,1),...
                            dataStruct(ii).t_r,dt,st);
                end

                % F_linear Variable key:
                % c(1) = Glassy Creep Compliance (Jg)
                % c(2) = First Compliance (J1)
                % c(3) = First Retardation Time (t_R_1)
                % c(4) = Second Compliance (J2)
                % c(5) = Second Retardation Time (t_R_2)
                % c(6) = Third Compliance (J3)
                % c(7) = Third Retardation Time (t_R_3)
                % c(8) = Fourth Compliance (J4)
                % c(9) = Fourth Retardation Time (t_R_4)
                % c(10) = Fifth Compliance (J5)
                % c(11) = Fifth Retardation Time (t_R_5)
                % inputs(:,1) = dataStruct(ii).t_r
                % Y = dataStruct(ii).chi_linear_log

                % Check the performance
                scatter(x_lin,y_lin);
                hold on
                title('Linear Force Fit')
                xlabel('Time [s]')
                ylabel('Force [N]')
                loglog(x_lin,F_linFit(dataStruct(ii).Fdot,x_lin),'r');
                set(gca,'xscale','log')
                set(gca,'yscale','log')
                grid on
                ylim([0.9*min(y_lin) max(y_lin)*1.1])
                xlim([0.9*min(x_lin) max(x_lin)*1.1])
                legendString{ii} = sprintf('Point %d, v_{approach} = %1.4g',point_number(ii),v_approach(ii));
                legend({[legendString{ii} ' Data'],'Linear Fdot Fit'},'Location','southoutside',...
                    'Orientation','horizontal')
                hold off
                
                % Uncomment below to save the linear figure
%                 saveas(gcf, sprintf('TempFigs/LinFit-nVoigt_%d-Point_%d-RunNo_%d',voigtArray(iVoigt),point_number(ii),run_number(ii)), 'jpg')

                clearvars x_lin y_lin beta_dist beta0_dist resnorm
                
                
                % Multiple-Voigt-Branch Nonlinear Fitting using the Convolution Method
                % Change to Logarithmic Scaling to use Enrique et al.'s Fit Method
                dt = (dataStruct(ii).dt);
                st = dataStruct(ii).t_r(end);
                h_norm = ((16.0*sqrt(r_tip))/3).*dataStruct(ii).h_r.^1.5;
                
                % Initialize arrays
                F_log = [];
                t_log = [];
                h_log = [];
                h_norm_log = [];
                
                % Get logarithmic data streams for analysis
                F_log = log_scale(dataStruct(ii).F_r,dataStruct(ii).t_r,dt,st);
                t_log = log_scale(dataStruct(ii).t_r,dataStruct(ii).t_r,dt,st);
                h_log = log_scale(dataStruct(ii).h_r,dataStruct(ii).t_r,dt,st);
                
                % Save the logarithmic indentation to the structure
                dataStruct(ii).h_log = h_log;
                h_norm_log = log_scale(h_norm,dataStruct(ii).t_r,dt,st);
                
                % Save the normalized indentation (per the Lopez et al.
                % methodology)
                dataStruct(ii).h_norm_log = h_norm_log;
                
                % Create the subref function (for isolating the subset of
                % the convolution).
                subref = @(A) A(1:size(dataStruct(ii).t_r,1),1);

                % Function Shape for Multiple Voigt with No Load Assumption
                switch n_terms
                    case 1
                        F_conv = @(c,inputs) convnfft(( (c(2)/c(3)*exp(-inputs(:,1)/c(3))) ),...
                            inputs(:,2),'full');
                        F_conv_wrapper = @(c,inputs) log_scale(...
                            subref(F_conv(c,inputs))*dt + c(1)*inputs(:,2),...
                            dataStruct(ii).t_r,dt,st);

                        lb = lb_all(1:3);
                        ub = ub_all(1:3);
                        initParams = initParams_all(1:3);

                    case 2
                        F_conv = @(c,inputs) convnfft(( (c(2)/c(3)*exp(-inputs(:,1)/c(3)))...
                            + (c(4)/c(5)*exp(-inputs(:,1)/c(5))) ),inputs(:,2),'full');
                        F_conv_wrapper = @(c,inputs) log_scale(...
                            subref(F_conv(c,inputs))*dt + c(1)*inputs(:,2),...
                            dataStruct(ii).t_r,dt,st);

                        lb = lb_all(1:5);
                        ub = ub_all(1:5);
                        initParams = initParams_all(1:5);

                    case 3
                        F_conv = @(c,inputs) convnfft(( (c(2)/c(3)*exp(-inputs(:,1)/c(3)))...
                            + (c(4)/c(5)*exp(-inputs(:,1)/c(5)))...
                            + (c(6)/c(7)*exp(-inputs(:,1)/c(7))) ),inputs(:,2),'full');
                        F_conv_wrapper = @(c,inputs) log_scale(...
                            subref(F_conv(c,inputs))*dt + c(1)*inputs(:,2),...
                            dataStruct(ii).t_r,dt,st);

                        lb = lb_all(1:7);
                        ub = ub_all(1:7);
                        initParams = initParams_all(1:7);

                    case 4
                        F_conv = @(c,inputs) convnfft(( (c(2)/c(3)*exp(-inputs(:,1)/c(3)))...
                            + (c(4)/c(5)*exp(-inputs(:,1)/c(5)))...
                            + (c(6)/c(7)*exp(-inputs(:,1)/c(7)))...
                            + (c(8)/c(9)*exp(-inputs(:,1)/c(9))) ),inputs(:,2),'full');
                        F_conv_wrapper = @(c,inputs) log_scale(...
                            subref(F_conv(c,inputs))*dt + c(1)*inputs(:,2),...
                            dataStruct(ii).t_r,dt,st);

                        lb = lb_all(1:9);
                        ub = ub_all(1:9);
                        initParams = initParams_all(1:9);

                    case 5
                        F_conv = @(c,inputs) convnfft(( (c(2)/c(3)*exp(-inputs(:,1)/c(3)))...
                            + (c(4)/c(5)*exp(-inputs(:,1)/c(5)))...
                            + (c(6)/c(7)*exp(-inputs(:,1)/c(7)))...
                            + (c(8)/c(9)*exp(-inputs(:,1)/c(9)))...
                            + (c(10)/c(11)*exp(-inputs(:,1)/c(11))) ),inputs(:,2),'full');
                        F_conv_wrapper = @(c,inputs) log_scale(...
                            subref(F_conv(c,inputs))*dt + c(1)*inputs(:,2),...
                            dataStruct(ii).t_r,dt,st);

                        lb = lb_all(1:11);
                        ub = ub_all(1:11);
                        initParams = initParams_all(1:11);

                end

                % F_conv Variable key:
                % c(1) = Glassy Creep Compliance (Jg)
                % c(2) = First Compliance (J1)
                % c(3) = First Retardation Time (t_R_1)
                % c(4) = Second Compliance (J2)
                % c(5) = Second Retardation Time (t_R_2)
                % c(6) = Third Compliance (J3)
                % c(7) = Third Retardation Time (t_R_3)
                % c(8) = Fourth Compliance (J4)
                % c(9) = Fourth Retardation Time (t_R_4)
                % c(10) = Fifth Compliance (J5)
                % c(11) = Fifth Retardation Time (t_R_5)
                % inputs(:,1) = dataStruct(ii).t_r
                % inputs(:,2) = dataStruct(ii).F_r
                % Y = h_norm_log
                
                % Change n_samples to alter the number of times a fit
                % occurs per number of voigt terms.
                n_samples = 1e3;
                
                % Initialize arrays for parallel computation
                beta_dist = zeros(length(lb),n_samples);
                beta0_dist = zeros(size(beta_dist));
                resnorm_dist = zeros(1,n_samples);
                beta_dist_linear = zeros(length(lb),n_samples);
                beta0_dist_linear = zeros(size(beta_dist_linear));
                resnorm_dist_linear = zeros(1,n_samples);
                
                % Set the X and Y data
                x_fit = [dataStruct(ii).t_r, dataStruct(ii).F_r];
                y_fit = h_norm_log;
                y_fit_linear = dataStruct(ii).chi_linear_log;

                progressString = sprintf('Convolution Approach, Curve #%d\nInvestigating Material Parameters\nParallel Search Running...',ii);
                hbar = parfor_progressbar(n_samples,progressString);
                warning('off');

                % Parallel Loop for Random Search
                parfor i = 1:n_samples
                    try
                        % Create random starting points within the upper
                        % and lower bounds.
                        beta0 = ((ub-lb).*rand(size(initParams)) + lb);

                        % Perform the fit using lsqcurvefit
                        options = optimoptions('lsqcurvefit','Algorithm','trust-region-reflective',...
                            'MaxFunctionEvaluations', 1.5e4,...
                            'MaxIterations', 1e4,...
                            'FiniteDifferenceType','central',...
                            'FunctionTolerance', 0,...
                            'OptimalityTolerance',scaling,...
                            'StepTolerance', scaling,...
                            'Display', 'none');

                        [betatemplsq,resnormlsq,residuallsq,exitflaglsq,outputlsq,lambdalsq,jacobianlsq] = ...
                            lsqcurvefit(F_conv_wrapper,beta0,x_fit,y_fit,lb,ub,options);
                        
                        % Save the final parameters, initial guesses, and
                        % normalized residual array.
                        beta_dist(:,i) = betatemp';
                        beta0_dist(:,i) = beta0;
                        resnorm_dist(i) = resnorm;
                        
                        % Clear the temporary variables for use again
                        betatemp = [];
                        resnorm = [];
                        residual = [];
                        exitflag = [];
                        output = [];
                        lambda = [];
                        jacobian = [];
                    
                        % Perform the Linear fit using lsqcurvefit
                        options = optimoptions('lsqcurvefit','Algorithm','interior-point',...
                            'MaxFunctionEvaluations', 1e3,...
                            'MaxIterations', 1e3,...
                            'FiniteDifferenceType','central',...
                            'FunctionTolerance', scaling,...
                            'OptimalityTolerance', scaling,...
                            'StepTolerance', scaling,...
                            'Display', 'none');

                        [betatemp,resnorm,residual,exitflag,output,lambda,jacobian] = ...
                            lsqcurvefit(F_linear_fit,beta0,x_fit(:,1),...
                            y_fit_linear,lb,ub,options);
                        
                        % Save the final parameters, initial guesses, and
                        % normalized residual
                        beta_dist_linear(:,i) = betatemp';
                        beta0_dist_linear(:,i) = beta0;
                        resnorm_dist_linear(i) = resnorm;

                        betatemp = [];
                        resnorm = [];
                        residual = [];
                        exitflag = [];
                        output = [];
                        lambda = [];
                        jacobian = [];

                    catch
                        
                        % If it didn't work, insert NaNs to be filtered
                        % later on.
                        beta_dist(:,i) = NaN;
                        beta0_dist(:,i) = NaN;
                        resnorm_dist(i) = NaN;

                        beta_dist_linear(:,i) = NaN;
                        beta0_dist_linear(:,i) = NaN;
                        resnorm_dist_linear(i) = NaN;
                        
                    end
                    
                        hbar.iterate(1) % Increase progressbar by 1 iteration
                
                end

                close(hbar)
                warning('on');
                
                % Find the best set of linear parameters
                [minRes,idx_linear] = min(resnorm_dist_linear);
                dataStruct(ii).linearParams = beta_dist_linear(:,idx_linear);
                
                minRes = [];
                
                % Find the best convolution parameters
                [minRes,idx] = min(resnorm_dist);
                dataStruct(ii).convParams = beta_dist(:,idx);
                
                % Also save the top ten parameter sets
                [B,I] = sort(resnorm_dist);
                dataStruct(ii).convParams_alternatives = beta_dist(:,I(1:10));

                minRes = [];
                B = [];

                fprintf(fid,'\r\n\nThe Lowest Normal Residual (resnorm) for the Convolution fit was %3.4e, using:\r\n',minRes);
                
                % Write the parameters found to the file being created.
                switch n_terms
                    case 1
                        fprintf(fid,'\r\n\r\nFitting Params (Lopez et al Method)\r\n');
                        fprintf(fid,'J_g = %3.6g\r\n',beta_dist(1,idx));
                        fprintf(fid,'J_1 = %3.6g, tau_1 = %3.6g\r\n',beta_dist(2,idx),beta_dist(3,idx));

                        % Calculate the Fluidity (chi)
                        dataStruct(length(Files)+kk).chi_conv = ( ((beta_dist(2,idx)/beta_dist(3,idx)*exp(-x_fit(:,1)/beta_dist(3,idx))))...
                            + beta_dist(1,idx) );
                    case 2
                        fprintf(fid,'\r\n\r\nFitting Params (Lopez et al Method)\r\n');
                        fprintf(fid,'J_g = %3.6g\r\n',beta_dist(1,idx));
                        fprintf(fid,'J_1 = %3.6g, tau_1 = %3.6g\r\n',beta_dist(2,idx),beta_dist(3,idx));
                        fprintf(fid,'J_2 = %3.6g, tau_2 = %3.6g\r\n',beta_dist(4,idx),beta_dist(5,idx));

                        % Calculate the Fluidity (chi)
                        dataStruct(length(Files)+kk).chi_conv = ( ((beta_dist(2,idx)/beta_dist(3,idx)*exp(-x_fit(:,1)/beta_dist(3,idx)))...
                            + (beta_dist(4,idx)/beta_dist(5,idx)*exp(-x_fit(:,1)/beta_dist(5,idx)))) + beta_dist(1,idx) );
                    case 3
                        fprintf(fid,'\r\n\r\nFitting Params (Lopez et al Method)\r\n');
                        fprintf(fid,'J_g = %3.6g\r\n',beta_dist(1,idx));
                        fprintf(fid,'J_1 = %3.6g, tau_1 = %3.6g\r\n',beta_dist(2,idx),beta_dist(3,idx));
                        fprintf(fid,'J_2 = %3.6g, tau_2 = %3.6g\r\n',beta_dist(4,idx),beta_dist(5,idx));
                        fprintf(fid,'J_3 = %3.6g, tau_3 = %3.6g\r\n',beta_dist(6,idx),beta_dist(7,idx));

                        % Calculate the Fluidity (chi)
                        dataStruct(length(Files)+kk).chi_conv = ( ((beta_dist(2,idx)/beta_dist(3,idx)*exp(-x_fit(:,1)/beta_dist(3,idx)))...
                            + (beta_dist(4,idx)/beta_dist(5,idx)*exp(-x_fit(:,1)/beta_dist(5,idx)))...
                            + (beta_dist(6,idx)/beta_dist(7,idx)*exp(-x_fit(:,1)/beta_dist(7,idx)))) + beta_dist(1,idx) );
                    case 4
                        fprintf(fid,'\r\n\r\nFitting Params (Lopez et al Method)\r\n');
                        fprintf(fid,'J_g = %3.6g\r\n',beta_dist(1,idx));
                        fprintf(fid,'J_1 = %3.6g, tau_1 = %3.6g\r\n',beta_dist(2,idx),beta_dist(3,idx));
                        fprintf(fid,'J_2 = %3.6g, tau_2 = %3.6g\r\n',beta_dist(4,idx),beta_dist(5,idx));
                        fprintf(fid,'J_3 = %3.6g, tau_3 = %3.6g\r\n',beta_dist(6,idx),beta_dist(7,idx));
                        fprintf(fid,'J_4 = %3.6g, tau_4 = %3.6g\r\n',beta_dist(8,idx),beta_dist(9,idx));

                        % Calculate the Fluidity (chi)
                        dataStruct(length(Files)+kk).chi_conv = ( ((beta_dist(2,idx)/beta_dist(3,idx)*exp(-x_fit(:,1)/beta_dist(3,idx)))...
                            + (beta_dist(4,idx)/beta_dist(5,idx)*exp(-x_fit(:,1)/beta_dist(5,idx)))...
                            + (beta_dist(6,idx)/beta_dist(7,idx)*exp(-x_fit(:,1)/beta_dist(7,idx)))...
                            + (beta_dist(8,idx)/beta_dist(9,idx)*exp(-x_fit(:,1)/beta_dist(9,idx)))) + beta_dist(1,idx) );
                    case 5
                        fprintf(fid,'\r\n\r\nFitting Params (Lopez et al Method)\r\n');
                        fprintf(fid,'J_g = %3.6g\r\n',beta_dist(1,idx));
                        fprintf(fid,'J_1 = %3.6g, tau_1 = %3.6g\r\n',beta_dist(2,idx),beta_dist(3,idx));
                        fprintf(fid,'J_2 = %3.6g, tau_2 = %3.6g\r\n',beta_dist(4,idx),beta_dist(5,idx));
                        fprintf(fid,'J_3 = %3.6g, tau_3 = %3.6g\r\n',beta_dist(6,idx),beta_dist(7,idx));
                        fprintf(fid,'J_4 = %3.6g, tau_4 = %3.6g\r\n',beta_dist(8,idx),beta_dist(9,idx));
                        fprintf(fid,'J_5 = %3.6g, tau_5 = %3.6g\r\n',beta_dist(10,idx),beta_dist(11,idx));

                        % Calculate the Fluidity (chi)
                        dataStruct(length(Files)+kk).chi_conv = ( ((beta_dist(2,idx)/beta_dist(3,idx)*exp(-x_fit(:,1)/beta_dist(3,idx)))...
                            + (beta_dist(4,idx)/beta_dist(5,idx)*exp(-x_fit(:,1)/beta_dist(5,idx)))...
                            + (beta_dist(6,idx)/beta_dist(7,idx)*exp(-x_fit(:,1)/beta_dist(7,idx)))...
                            + (beta_dist(8,idx)/beta_dist(9,idx)*exp(-x_fit(:,1)/beta_dist(9,idx)))...
                            + (beta_dist(10,idx)/beta_dist(11,idx)*exp(-x_fit(:,1)/beta_dist(11,idx)))) + beta_dist(1,idx) );
                end

                % Check the performance
                scatter(t_log,y_fit,'r');
                hold on
                scatter(t_log,y_fit_area,'b');
                set(gca,'xscale','log')
                set(gca,'yscale','log')
                title('Convolution-Method Force Fit')
                xlabel('Time [s]')
                ylabel('Force [N]')
                loglog(t_log,F_conv_wrapper(beta_dist(:,idx),[dataStruct(ii).t_r, dataStruct(ii).F_r]),'r');
                grid on
                legendString{ii} = sprintf('Point %d, v_{approach} = %1.4g',point_number(ii),v_approach(ii));
                tempString = sprintf('Convolution Fit, %d-Terms',n_terms);
                legend({[legendString{ii} ' Data'],tempString},'Location','southoutside',...
                    'Orientation','horizontal')
                hold off

                saveas(gcf, sprintf('TempFigs/ConvFit-nVoigt_%d-Point_%d-RunNo_%d',voigtArray(iVoigt),point_number(ii),run_number(ii)), 'jpg')

                if saveMats == 'y'
                    save([pathname '\' sprintf('fitting-workspaceVariables-FileNo_%d.mat',ii)])
                end

            end

        case 'a'
            
            % Averaged data analysis
            
            % Loop for unique velocities found in the folder
            for kk = 1:length(v_unique)
                fprintf(fid,'\r\n\r\nAveraged File Analysis, Load Level %d\r\n=================\r\n',kk);

                x_lin = dataStruct(length(Files)+kk).t_r_log;               % X data for linear fit
                y_lin = dataStruct(length(Files)+kk).F_r_log;               % Y data for linear fit

                init_params = polyfit(x_lin, y_lin, 1);                     % Initial Linear Fit Parameters (guess)

                % Linear Fit Model Definition
                F_linFit = @(c,input) c .* input;

                lb = 0;                                                     % Zero Newtons/s is the minimum force rate possible
                ub = 1e4;                                                   % 1e4 Newton/s is the upper bound
                
                % Settings for the linear parameter fit
                options = optimoptions('lsqcurvefit','Algorithm','trust-region-reflective',...
                    'MaxFunctionEvaluations', 1e3,...
                    'MaxIterations', 1e3,...
                    'FiniteDifferenceType','central',...
                    'FunctionTolerance', scaling,...
                    'OptimalityTolerance', scaling,...
                    'StepTolerance', scaling,...
                    'Display', 'none');
                
                % n_samples will control the number of fit attempts for
                % this loop.
                n_samples = 1e3;
                
                % Initialize arrays
                beta_dist = zeros(length(lb),n_samples);
                beta0_dist = zeros(size(beta_dist));
                resnorm_dist = zeros(1,n_samples);

                progressString = sprintf('Linear Force Assumption, Averaged Curve\nInvestigating Fdot\nParallel Search Running...');
                hbar = parfor_progressbar(n_samples,progressString);
                warning('off');

                % Loop for linear parameter grid search

                % Turned off parallel. To turn on, replace 'for' with 
                % 'parfor' on the next line.
                for i = 1:n_samples
                    try
                        beta0 = init_params(1);
                        [betatemp,resnorm,residual,exitflag,output,lambda,jacobian] = ...
                            lsqcurvefit(F_linFit,beta0,x_lin,y_lin,lb,ub,options);

                        beta_dist(:,i) = betatemp;
                        beta0_dist(:,i) = beta0;
                        resnorm_dist(i) = sum(abs(residual));
                    catch
                        beta_dist(:,i) = NaN;
                        beta0_dist(:,i) = NaN;
                        resnorm_dist(i) = NaN;
                    end
                        hbar.iterate(1) % Increase progressbar by 1 iteration
                end

                close(hbar)
                warning('on');

                [minRes,idx] = min(resnorm_dist);
                fprintf(fid,'\r\n\nThe Lowest Normal Residual (resnorm) for the Linear fit was %3.4e, using:\r\n',minRes);
                fprintf(fid,'Fdot = %3.6g\r\n',beta_dist(1,idx));
                dataStruct(length(Files)+kk).Fdot = beta_dist(1,idx);

                % Calculate the Linear Fluidity (chi)
                % According to Eq. 14 in Lopez et al., This is the relation 
                % between chi and tip location when force is assumed to be 
                % linear

                dt = (dataStruct(length(Files)+kk).dt);
                st = dataStruct(length(Files)+kk).t_r(end);

                dataStruct(length(Files)+kk).chi_linear = ((16.0*sqrt(r_tip))/3)...
                    .* (dataStruct(length(Files)+kk).h_r.^1.5) ./ dataStruct(length(Files)+kk).Fdot;
                dataStruct(length(Files)+kk).chi_linear_log = log_scale(dataStruct(length(Files)+kk).chi_linear,...
                    dataStruct(length(Files)+kk).t_r, dt, st);

                % Function Shape for Multiple Voigt with Linear Load Assumption
                switch n_terms
                    case 1
                        F_linear_fit = @(c,inputs) log_scale(( c(2).*c(3).*(exp(-inputs(:,1)./c(3))-1) )...
                            + (c(1)+c(2))*inputs(:,1),...
                            dataStruct(length(Files)+kk).t_r,dt,st);

                    case 2
                        F_linear_fit = @(c,inputs) log_scale(( c(2).*c(3).*(exp(-inputs(:,1)./c(3))-1)...
                            + c(4).*c(5).*(exp(-inputs(:,1)./c(5))-1) )...
                            + (c(1)+c(2)+c(4))*inputs(:,1),...
                            dataStruct(length(Files)+kk).t_r,dt,st);

                    case 3
                        F_linear_fit = @(c,inputs) log_scale(( c(2).*c(3).*(exp(-inputs(:,1)./c(3))-1)...
                            + c(4).*c(5).*(exp(-inputs(:,1)./c(5))-1)...
                            + c(6).*c(7).*(exp(-inputs(:,1)./c(7))-1) )...
                            + (c(1)+c(2)+c(4)+c(6))*inputs(:,1),...
                            dataStruct(length(Files)+kk).t_r,dt,st);

                    case 4
                        F_linear_fit = @(c,inputs) log_scale(( c(2).*c(3).*(exp(-inputs(:,1)./c(3))-1)...
                            + c(4).*c(5).*(exp(-inputs(:,1)./c(5))-1)...
                            + c(6).*c(7).*(exp(-inputs(:,1)./c(7))-1)...
                            + c(8).*c(9).*(exp(-inputs(:,1)./c(9))-1) )...
                            + (c(1)+c(2)+c(4)+c(6)+c(8))*inputs(:,1),...
                            dataStruct(length(Files)+kk).t_r,dt,st);

                    case 5
                        F_linear_fit = @(c,inputs) log_scale(( c(2).*c(3).*(exp(-inputs(:,1)./c(3))-1)...
                            + c(4).*c(5).*(exp(-inputs(:,1)./c(5))-1)...
                            + c(6).*c(7).*(exp(-inputs(:,1)./c(7))-1)...
                            + c(8).*c(9).*(exp(-inputs(:,1)./c(9))-1)...
                            + c(10).*c(11).*(exp(-inputs(:,1)./c(11))-1) )...
                            + (c(1)+c(2)+c(4)+c(6)+c(8)+c(10))*inputs(:,1),...
                            dataStruct(length(Files)+kk).t_r,dt,st);
                end

                % F_linear Variable key:
                % c(1) = Glassy Creep Compliance (Jg)
                % c(2) = First Compliance (J1)
                % c(3) = First Retardation Time (t_R_1)
                % c(4) = Second Compliance (J2)
                % c(5) = Second Retardation Time (t_R_2)
                % c(6) = Third Compliance (J3)
                % c(7) = Third Retardation Time (t_R_3)
                % c(8) = Fourth Compliance (J4)
                % c(9) = Fourth Retardation Time (t_R_4)
                % c(10) = Fifth Compliance (J5)
                % c(11) = Fifth Retardation Time (t_R_5)
                % inputs(:,1) = dataStruct(length(Files)+kk).t_r
                % Y = dataStruct(length(Files)+kk).chi_linear_log

                % Check the performance
                scatter(x_lin,y_lin);
                hold on
                title('Linear Force Fit')
                xlabel('Time [s]')
                ylabel('Force [N]')
                loglog(x_lin,F_linFit(dataStruct(length(Files)+kk).Fdot,x_lin),'r');
                set(gca,'xscale','log')
                set(gca,'yscale','log')
                grid on
                ylim([0.9*min(y_lin) max(y_lin)*1.1])
                xlim([0.9*min(x_lin) max(x_lin)*1.1])
                legend({'Averaged Curve Data','Linear Fdot Fit'},'Location','southoutside',...
                    'Orientation','horizontal')
                hold off

                % Uncomment the line below to save the linear estimation.
%                 saveas(gcf, sprintf('TempFigs/Averaged-LinFit-nVoigt_%d',voigtArray(iVoigt)), 'jpg')

                clearvars x_lin y_lin beta_dist beta0_dist resnorm

                % Multiple-Voigt-Branch Nonlinear Fitting using the Convolution Method

                % Change to Logarithmic Scaling to use Lopez et al.'s Fit Method
                dt = (dataStruct(length(Files)+kk).dt);
                st = dataStruct(length(Files)+kk).t_r(end);
                h_norm = ((16.0*sqrt(r_tip))/3).*dataStruct(length(Files)+kk).h_r.^1.5;
                
                % Initialize the log arrays
                F_log = [];
                t_log = [];
                h_log = [];
                h_norm_log = [];
                
                % Generate the log datasets for fitting
                F_log = log_scale(dataStruct(length(Files)+kk).F_r,dataStruct(length(Files)+kk).t_r,dt,st);
                t_log = log_scale(dataStruct(length(Files)+kk).t_r,dataStruct(length(Files)+kk).t_r,dt,st);
                h_log = log_scale(dataStruct(length(Files)+kk).h_r,dataStruct(length(Files)+kk).t_r,dt,st);
                
                % Save the indentation (normalized/non) log datasets
                dataStruct(length(Files)+kk).h_log = h_log;
                h_norm_log = log_scale(h_norm,dataStruct(length(Files)+kk).t_r,dt,st);
                dataStruct(length(Files)+kk).h_norm_log = h_norm_log;
                
                % Create the subref function for convolutions
                subref = @(A) A(1:size(dataStruct(length(Files)+kk).t_r,1),1);

                % Function Shape for Multiple Voigt with No Load Assumption
                switch n_terms
                    case 1
                        F_conv = @(c,inputs) convnfft(( (c(2)/c(3)*exp(-inputs(:,1)/c(3))) ),...
                            inputs(:,2),'full');
                        F_conv_wrapper = @(c,inputs) log_scale(...
                            subref(F_conv(c,inputs))*dt + c(1)*inputs(:,2),...
                            dataStruct(length(Files)+kk).t_r,dt,st);

                        lb = lb_all(1:3);
                        ub = ub_all(1:3);
                        initParams = initParams_all(1:3);

                    case 2
                        F_conv = @(c,inputs) convnfft(( (c(2)/c(3)*exp(-inputs(:,1)/c(3)))...
                            + (c(4)/c(5)*exp(-inputs(:,1)/c(5))) ),inputs(:,2),'full');
                        F_conv_wrapper = @(c,inputs) log_scale(...
                            subref(F_conv(c,inputs))*dt + c(1)*inputs(:,2),...
                            dataStruct(length(Files)+kk).t_r,dt,st);

                        lb = lb_all(1:5);
                        ub = ub_all(1:5);
                        initParams = initParams_all(1:5);

                    case 3
                        F_conv = @(c,inputs) convnfft(( (c(2)/c(3)*exp(-inputs(:,1)/c(3)))...
                            + (c(4)/c(5)*exp(-inputs(:,1)/c(5)))...
                            + (c(6)/c(7)*exp(-inputs(:,1)/c(7))) ),inputs(:,2),'full');
                        F_conv_wrapper = @(c,inputs) log_scale(...
                            subref(F_conv(c,inputs))*dt + c(1)*inputs(:,2),...
                            dataStruct(length(Files)+kk).t_r,dt,st);

                        lb = lb_all(1:7);
                        ub = ub_all(1:7);
                        initParams = initParams_all(1:7);

                    case 4
                        F_conv = @(c,inputs) convnfft(( (c(2)/c(3)*exp(-inputs(:,1)/c(3)))...
                            + (c(4)/c(5)*exp(-inputs(:,1)/c(5)))...
                            + (c(6)/c(7)*exp(-inputs(:,1)/c(7)))...
                            + (c(8)/c(9)*exp(-inputs(:,1)/c(9))) ),inputs(:,2),'full');
                        F_conv_wrapper = @(c,inputs) log_scale(...
                            subref(F_conv(c,inputs))*dt + c(1)*inputs(:,2),...
                            dataStruct(length(Files)+kk).t_r,dt,st);

                        lb = lb_all(1:9);
                        ub = ub_all(1:9);
                        initParams = initParams_all(1:9);

                    case 5
                        F_conv = @(c,inputs) convnfft(( (c(2)/c(3)*exp(-inputs(:,1)/c(3)))...
                            + (c(4)/c(5)*exp(-inputs(:,1)/c(5)))...
                            + (c(6)/c(7)*exp(-inputs(:,1)/c(7)))...
                            + (c(8)/c(9)*exp(-inputs(:,1)/c(9)))...
                            + (c(10)/c(11)*exp(-inputs(:,1)/c(11))) ),inputs(:,2),'full');

                        F_conv_wrapper = @(c,inputs) log_scale(...
                            subref(F_conv(c,inputs))*dt + c(1)*inputs(:,2),...
                            dataStruct(length(Files)+kk).t_r,dt,st);

                        lb = lb_all(1:11);
                        ub = ub_all(1:11);
                        initParams = initParams_all(1:11);
                end

                % F_conv Variable key:
                % c(1) = Glassy Creep Compliance (Jg)
                % c(2) = First Compliance (J1)
                % c(3) = First Retardation Time (t_R_1)
                % c(4) = Second Compliance (J2)
                % c(5) = Second Retardation Time (t_R_2)
                % c(6) = Third Compliance (J3)
                % c(7) = Third Retardation Time (t_R_3)
                % c(8) = Fourth Compliance (J4)
                % c(9) = Fourth Retardation Time (t_R_4)
                % c(10) = Fifth Compliance (J5)
                % c(11) = Fifth Retardation Time (t_R_5)
                % inputs(:,1) = dataStruct(length(Files)+kk).t_r
                % inputs(:,2) = dataStruct(length(Files)+kk).F_r
                % Y = h_norm_log
                
                % Change n_samples to alter the number of fit attempts
                % performed for this loop's number of voigt parameters.
                n_samples = 1e3;
                
                % Initialize arrays
                beta_dist = zeros(length(lb),n_samples);
                beta0_dist = zeros(size(beta_dist));
                resnorm_dist = zeros(1,n_samples);
                beta_dist_linear = zeros(length(lb),n_samples);
                beta0_dist_linear = zeros(size(beta_dist_linear));
                resnorm_dist_linear = zeros(1,n_samples);
                
                % Select the X and Y data for fitting
                x_fit = [dataStruct(length(Files)+kk).t_r, dataStruct(length(Files)+kk).F_r];
                y_fit = dataStruct(length(Files)+kk).h_norm_log;
                y_fit_linear = dataStruct(length(Files)+kk).chi_linear_log;

                progressString = sprintf('Convolution Approach, Averaged Curve\nInvestigating Material Parameters\nParallel Search Running...');
                hbar = parfor_progressbar(n_samples,progressString);
                warning('off');

                % Parallel Loop for Random Search
                parfor i = 1:n_samples
                    try
                        % Generate initial guesses
                        beta0 = ((ub-lb).*rand(size(initParams)) + lb);
                        beta0(1) = (1E-8 - 1E-10).*(rand());                % Glassy Compliance Guess
                        beta0(2:2:end) = (1E-8 - 1E-12).*(rand(size(beta0(2:2:end)))); % All other compliance guesses

                        % Perform the convolution fit using lsqcurvefit
                        options = optimoptions('lsqcurvefit','Algorithm','trust-region-reflective',...
                            'MaxFunctionEvaluations', 1.5e4,...
                            'MaxIterations', 1e4,...
                            'FiniteDifferenceType','central',...
                            'FunctionTolerance', 0,...
                            'OptimalityTolerance',scaling,...
                            'StepTolerance', scaling,...
                            'Display', 'none');

                        [betatemp,resnorm,residual,exitflag,output,lambda,jacobian] = ...
                            lsqcurvefit(F_conv_wrapper,beta0,x_fit,y_fit,lb,ub,options);
                        
                        % Save the final parameters, initial guesses, and
                        % normalized residual
                        beta_dist(:,i) = betatemp';
                        beta0_dist(:,i) = beta0;
                        resnorm_dist(i) = resnorm;

                        betatemp = [];
                        resnorm = [];
                        residual = [];
                        exitflag = [];
                        output = [];
                        lambda = [];
                        jacobian = [];

                        % Perform the Linear Fit using lsqcurvefit
                        options = optimoptions('lsqcurvefit','Algorithm','trust-region-reflective',...
                            'MaxFunctionEvaluations', 1e3,...
                            'MaxIterations', 1e3,...
                            'FiniteDifferenceType','central',...
                            'FunctionTolerance', scaling,...
                            'OptimalityTolerance', scaling,...
                            'StepTolerance', scaling,...
                            'Display', 'none');

                        [betatemp,resnorm,residual,exitflag,output,lambda,jacobian] = ...
                            lsqcurvefit(F_linear_fit,beta0,x_fit(:,1),...
                            y_fit_linear,lb,ub,options);
    
                        % Save the final parameters, initial guesses, and
                        % normalized residual.
                        beta_dist_linear(:,i) = betatemp';
                        beta0_dist_linear(:,i) = beta0;
                        resnorm_dist_linear(i) = resnorm;

                        betatemp = [];
                        resnorm = [];
                        residual = [];
                        exitflag = [];
                        output = [];
                        lambda = [];
                        jacobian = [];

                    catch
                        
                        % If the fit failed, insert NaNs for filtering
                        % later on.
                        beta_dist(:,i) = NaN;
                        beta0_dist(:,i) = NaN;
                        resnorm_dist(i) = NaN;

                        beta_dist_linear(:,i) = NaN;
                        beta0_dist_linear(:,i) = NaN;
                        resnorm_dist_linear(i) = NaN;

                        fprintf('\nEntered a NaN Row.\n');
                    end
                        hbar.iterate(1) % Increase progressbar by 1 iteration
                end

                close(hbar)
                warning('on');

                n_ranks = 10; % Number of ranks to save
                
                % Find the best linear parameter set
                [minRes,idx_linear] = min(resnorm_dist_linear);
                dataStruct(length(Files)+kk).linearParams = beta_dist_linear(:,idx_linear);

                minRes = [];
                
                % Find the best convolution parameter set
                [minRes,idx] = min(resnorm_dist);
                dataStruct(length(Files)+kk).convParams = beta_dist(:,idx);
                
                [B,I] = sort(resnorm_dist);
                dataStruct(length(Files)+kk).convParams_alternatives = beta_dist(:,I(1:n_ranks));
           
                % Print parameters to file
                switch n_terms
                    case 1
                        fprintf(fid,'\r\n\r\nFitting Params (Lopez et al Method)\r\n');
                        fprintf(fid,'J_g = %3.6g\r\n',beta_dist(1,idx));
                        fprintf(fid,'J_1 = %3.6g, tau_1 = %3.6g\r\n',beta_dist(2,idx),beta_dist(3,idx));

                        % Calculate the Fluidity (chi)
                        dataStruct(length(Files)+kk).chi_conv = ( ((beta_dist(2,idx)/beta_dist(3,idx)*exp(-x_fit(:,1)/beta_dist(3,idx))))...
                            + beta_dist(1,idx) );
                    case 2
                        fprintf(fid,'\r\n\r\nFitting Params (Lopez et al Method)\r\n');
                        fprintf(fid,'J_g = %3.6g\r\n',beta_dist(1,idx));
                        fprintf(fid,'J_1 = %3.6g, tau_1 = %3.6g\r\n',beta_dist(2,idx),beta_dist(3,idx));
                        fprintf(fid,'J_2 = %3.6g, tau_2 = %3.6g\r\n',beta_dist(4,idx),beta_dist(5,idx));

                        % Calculate the Fluidity (chi)
                        dataStruct(length(Files)+kk).chi_conv = ( ((beta_dist(2,idx)/beta_dist(3,idx)*exp(-x_fit(:,1)/beta_dist(3,idx)))...
                            + (beta_dist(4,idx)/beta_dist(5,idx)*exp(-x_fit(:,1)/beta_dist(5,idx)))) + beta_dist(1,idx) );
                    case 3
                        fprintf(fid,'\r\n\r\nFitting Params (Lopez et al Method)\r\n');
                        fprintf(fid,'J_g = %3.6g\r\n',beta_dist(1,idx));
                        fprintf(fid,'J_1 = %3.6g, tau_1 = %3.6g\r\n',beta_dist(2,idx),beta_dist(3,idx));
                        fprintf(fid,'J_2 = %3.6g, tau_2 = %3.6g\r\n',beta_dist(4,idx),beta_dist(5,idx));
                        fprintf(fid,'J_3 = %3.6g, tau_3 = %3.6g\r\n',beta_dist(6,idx),beta_dist(7,idx));

                        % Calculate the Fluidity (chi)
                        dataStruct(length(Files)+kk).chi_conv = ( ((beta_dist(2,idx)/beta_dist(3,idx)*exp(-x_fit(:,1)/beta_dist(3,idx)))...
                            + (beta_dist(4,idx)/beta_dist(5,idx)*exp(-x_fit(:,1)/beta_dist(5,idx)))...
                            + (beta_dist(6,idx)/beta_dist(7,idx)*exp(-x_fit(:,1)/beta_dist(7,idx)))) + beta_dist(1,idx) );
                    case 4
                        fprintf(fid,'\r\n\r\nFitting Params (Lopez et al Method)\r\n');
                        fprintf(fid,'J_g = %3.6g\r\n',beta_dist(1,idx));
                        fprintf(fid,'J_1 = %3.6g, tau_1 = %3.6g\r\n',beta_dist(2,idx),beta_dist(3,idx));
                        fprintf(fid,'J_2 = %3.6g, tau_2 = %3.6g\r\n',beta_dist(4,idx),beta_dist(5,idx));
                        fprintf(fid,'J_3 = %3.6g, tau_3 = %3.6g\r\n',beta_dist(6,idx),beta_dist(7,idx));
                        fprintf(fid,'J_4 = %3.6g, tau_4 = %3.6g\r\n',beta_dist(8,idx),beta_dist(9,idx));

                        % Calculate the Fluidity (chi)
                        dataStruct(length(Files)+kk).chi_conv = ( ((beta_dist(2,idx)/beta_dist(3,idx)*exp(-x_fit(:,1)/beta_dist(3,idx)))...
                            + (beta_dist(4,idx)/beta_dist(5,idx)*exp(-x_fit(:,1)/beta_dist(5,idx)))...
                            + (beta_dist(6,idx)/beta_dist(7,idx)*exp(-x_fit(:,1)/beta_dist(7,idx)))...
                            + (beta_dist(8,idx)/beta_dist(9,idx)*exp(-x_fit(:,1)/beta_dist(9,idx)))) + beta_dist(1,idx) );
                    case 5
                        fprintf(fid,'\r\n\r\nFitting Params (Lopez et al Method)\r\n');
                        fprintf(fid,'J_g = %3.6g\r\n',beta_dist(1,idx));
                        fprintf(fid,'J_1 = %3.6g, tau_1 = %3.6g\r\n',beta_dist(2,idx),beta_dist(3,idx));
                        fprintf(fid,'J_2 = %3.6g, tau_2 = %3.6g\r\n',beta_dist(4,idx),beta_dist(5,idx));
                        fprintf(fid,'J_3 = %3.6g, tau_3 = %3.6g\r\n',beta_dist(6,idx),beta_dist(7,idx));
                        fprintf(fid,'J_4 = %3.6g, tau_4 = %3.6g\r\n',beta_dist(8,idx),beta_dist(9,idx));
                        fprintf(fid,'J_5 = %3.6g, tau_5 = %3.6g\r\n',beta_dist(10,idx),beta_dist(11,idx));

                        % Calculate the Fluidity (chi)
                        dataStruct(length(Files)+kk).chi_conv = ( ((beta_dist(2,idx)/beta_dist(3,idx)*exp(-x_fit(:,1)/beta_dist(3,idx)))...
                            + (beta_dist(4,idx)/beta_dist(5,idx)*exp(-x_fit(:,1)/beta_dist(5,idx)))...
                            + (beta_dist(6,idx)/beta_dist(7,idx)*exp(-x_fit(:,1)/beta_dist(7,idx)))...
                            + (beta_dist(8,idx)/beta_dist(9,idx)*exp(-x_fit(:,1)/beta_dist(9,idx)))...
                            + (beta_dist(10,idx)/beta_dist(11,idx)*exp(-x_fit(:,1)/beta_dist(11,idx)))) + beta_dist(1,idx) );
                end

                % Print alternate parameters
                fprintf(fid,'\r\n\r\nAlternative Fitting Parameters (2:10)\r\n\r\n');
                for alti = 2:n_ranks
                    fprintf(fid,' \r\n\r\nRank Number %d:',alti);
                    switch n_terms
                        case 1
                            fprintf(fid,'\r\n\r\nFitting Params (Lopez et al Method)\r\n');
                            fprintf(fid,'J_g = %3.6g\r\n',beta_dist(1,I(alti)));
                            fprintf(fid,'J_1 = %3.6g, tau_1 = %3.6g\r\n',beta_dist(2,I(alti)),beta_dist(3,I(alti)));

                        case 2
                            fprintf(fid,'\r\n\r\nFitting Params (Lopez et al Method)\r\n');
                            fprintf(fid,'J_g = %3.6g\r\n',beta_dist(1,I(alti)));
                            fprintf(fid,'J_1 = %3.6g, tau_1 = %3.6g\r\n',beta_dist(2,I(alti)),beta_dist(3,I(alti)));
                            fprintf(fid,'J_2 = %3.6g, tau_2 = %3.6g\r\n',beta_dist(4,I(alti)),beta_dist(5,I(alti)));

                        case 3
                            fprintf(fid,'\r\n\r\nFitting Params (Lopez et al Method)\r\n');
                            fprintf(fid,'J_g = %3.6g\r\n',beta_dist(1,I(alti)));
                            fprintf(fid,'J_1 = %3.6g, tau_1 = %3.6g\r\n',beta_dist(2,I(alti)),beta_dist(3,I(alti)));
                            fprintf(fid,'J_2 = %3.6g, tau_2 = %3.6g\r\n',beta_dist(4,I(alti)),beta_dist(5,I(alti)));
                            fprintf(fid,'J_3 = %3.6g, tau_3 = %3.6g\r\n',beta_dist(6,I(alti)),beta_dist(7,I(alti)));

                        case 4
                            fprintf(fid,'\r\n\r\nFitting Params (Lopez et al Method)\r\n');
                            fprintf(fid,'J_g = %3.6g\r\n',beta_dist(1,I(alti)));
                            fprintf(fid,'J_1 = %3.6g, tau_1 = %3.6g\r\n',beta_dist(2,I(alti)),beta_dist(3,I(alti)));
                            fprintf(fid,'J_2 = %3.6g, tau_2 = %3.6g\r\n',beta_dist(4,I(alti)),beta_dist(5,I(alti)));
                            fprintf(fid,'J_3 = %3.6g, tau_3 = %3.6g\r\n',beta_dist(6,I(alti)),beta_dist(7,I(alti)));
                            fprintf(fid,'J_4 = %3.6g, tau_4 = %3.6g\r\n',beta_dist(8,I(alti)),beta_dist(9,I(alti)));

                        case 5
                            fprintf(fid,'\r\n\r\nFitting Params (Lopez et al Method)\r\n');
                            fprintf(fid,'J_g = %3.6g\r\n',beta_dist(1,I(alti)));
                            fprintf(fid,'J_1 = %3.6g, tau_1 = %3.6g\r\n',beta_dist(2,I(alti)),beta_dist(3,I(alti)));
                            fprintf(fid,'J_2 = %3.6g, tau_2 = %3.6g\r\n',beta_dist(4,I(alti)),beta_dist(5,I(alti)));
                            fprintf(fid,'J_3 = %3.6g, tau_3 = %3.6g\r\n',beta_dist(6,I(alti)),beta_dist(7,I(alti)));
                            fprintf(fid,'J_4 = %3.6g, tau_4 = %3.6g\r\n',beta_dist(8,I(alti)),beta_dist(9,I(alti)));
                            fprintf(fid,'J_5 = %3.6g, tau_5 = %3.6g\r\n',beta_dist(10,I(alti)),beta_dist(11,I(alti)));

                    end
                end
            end

            if saveMats == 'y'
                save([pathname '\fittingAverage-workspaceVariables.mat'])
            end
    end

    fclose('all');

    % Memory management
    clearvars -except dataStruct defl_InVOLS F_conv F_conv_wrapper F_linear_fit F_linFit...
        Files k_cantilever legendString n_terms pathname point_number r_tip...
        run_number subref t_log v_approach v_unique avAnalysis scaling saveMats...
        nu_tip nu_sample E_tip nSims voigtArray iVoigt skipNums

    % Check the Fit Performance
    figure
    
    switch avAnalysis
        case 'i'
            for ii = 1:length(Files)
                dt = (dataStruct(ii).dt);
                st = dataStruct(ii).t_r(end);
                t_log = log_scale(dataStruct(ii).t_r,dataStruct(ii).t_r,dt,st);

                inputs = [dataStruct(ii).t_r, dataStruct(ii).F_r];

                % Redefine subref using the current file's t_r
                subref = @(A) A(1:size(dataStruct(ii).t_r,1),1);

                % Define the convolution fit dataset generation function
                data_fit = subref(F_conv(dataStruct(ii).convParams,inputs)).*dt...
                    + dataStruct(ii).convParams(1).*inputs(:,2);
                data_fit_log = log_scale(data_fit, dataStruct(ii).t_r, dt, st);

                % Define the linear convolution fit dataset generation function
                linear_fit = subref(F_conv(dataStruct(ii).linearParams,inputs)).*dt...
                    + dataStruct(ii).linearParams(1).*inputs(:,2);
                linear_fit_log = log_scale(linear_fit, dataStruct(ii).t_r, dt, st);

                % Plot the first figure using the AFM data, and both new datasets
                figure('position',[50 100 1200 600]);
                plot(t_log, dataStruct(ii).h_norm_log, 'k*', 'LineWidth', 3)
                hold on
                grid on
                plot(t_log, data_fit_log, 'c-', 'LineWidth', 5)
                plot(t_log, linear_fit_log, 'r--', 'LineWidth', 3)
                legend({'Experimental Data' 'Non-linear squares fit, Non Linear' 'Non-linear squares fit, Linear'}, 'Location', 'eastoutside', 'Orientation', 'vertical')
                set(gca,'xscale','log')
                set(gca,'yscale','log')
                title('Non-linear Squares Fit', 'FontSize', 16)
                xlabel('Time [s]', 'FontSize', 20)
                ylabel('$\int_{0}^{t} U(t-\xi) F(\xi) d\xi$ [$m^2$]', 'Interpreter', 'latex', 'FontSize', 16)
                hold off

%                 saveas(gcf, sprintf('TempFigs/ComplianceParams/dataFit-nVoigt_%d-Point_%d-RunNo_%d',voigtArray(iVoigt),point_number(ii),run_number(ii)), 'jpg')

                % Move on to plotting the loss angle
                % First, generate a frequency array in log scale
                de0 = 2.0.*pi.*(1./dataStruct(ii).t_r(end));
                maxi = 2.0.*pi.*(1./mean(diff(dataStruct(ii).t_r)));
                omega = log_tw(de0,maxi);

                % Second, use the convolution J parameters
                % Calculate J_storage
                Jstorage = J_storage(omega,dataStruct(ii).convParams);

                % Calculate J_loss
                Jloss = J_loss(omega,dataStruct(ii).convParams);

                % Calculate the loss angle and save these results to the
                % structure variable
                theta = atan((Jloss./Jstorage).^-1).*(180/pi);

                dataStruct(ii).Jloss_conv = Jloss;
                dataStruct(ii).Jstorage_conv = Jstorage;
                dataStruct(ii).theta_conv = theta;

                % Lastly, the Linear J parameters
                % Calculate J_storage
                Jstorage_linear = J_storage(omega,dataStruct(ii).linearParams);

                % Calculate J_loss
                Jloss_linear = J_loss(omega,dataStruct(ii).linearParams);

                % Finally, calculate the loss angle and save these results to the
                % structure variable
                theta_linear = atan((Jloss_linear./Jstorage_linear).^-1).*(180/pi);

                dataStruct(ii).Jloss_linear = Jloss_linear;
                dataStruct(ii).Jstorage_linear = Jstorage_linear;
                dataStruct(ii).theta_linear = theta_linear;

                figure('position',[70 150 800 600]);    
                loglog(omega, theta, 'c', 'LineWidth', 5)
                hold on
                grid on
                loglog(omega, theta_linear, 'r--', 'LineWidth', 5)
                legend({'Fit Non Linear' 'Fit Linear'})
                xlabel('$\omega, \,rad/s$', 'FontSize', 16, 'Interpreter', 'Latex')
                ylabel('$\delta(\omega), \,deg$', 'FontSize', 16, 'Interpreter', 'Latex')
                xlim([10 10^4])
                hold off

%                 saveas(gcf, sprintf('TempFigs/ComplianceParams/lossAngle-nVoigt_%d-Point_%d-RunNo_%d',voigtArray(iVoigt),point_number(ii),run_number(ii)), 'jpg')
            end

            %close all

        case 'a'
            for kk = 1:length(v_unique)
                dt = (dataStruct(length(Files)+kk).dt);
                st = dataStruct(length(Files)+kk).t_r(end);
                t_log = log_scale(dataStruct(length(Files)+kk).t_r,dataStruct(length(Files)+kk).t_r,dt,st);

                inputs = [dataStruct(length(Files)+kk).t_r, dataStruct(length(Files)+kk).F_r];

                % Redefine subref using the current file's t_r
                subref = @(A) A(1:size(dataStruct(length(Files)+kk).t_r,1),1);

                % Define the convolution fit dataset generation function
                data_fit = subref(F_conv(dataStruct(length(Files)+kk).convParams,inputs)).*dt...
                    + dataStruct(length(Files)+kk).convParams(1).*inputs(:,2);
                data_fit_log = log_scale(data_fit, dataStruct(length(Files)+kk).t_r, dt, st);

                % Define the linear convolution fit dataset generation function
                linear_fit = subref(F_conv(dataStruct(length(Files)+kk).linearParams,inputs)).*dt...
                    + dataStruct(length(Files)+kk).linearParams(1).*inputs(:,2);
                linear_fit_log = log_scale(linear_fit, dataStruct(length(Files)+kk).t_r, dt, st);

                % Plot the first figure using the AFM data, and both new datasets
                figure('position',[50 100 1200 600]);
                plot(t_log, dataStruct(length(Files)+kk).h_norm_log, 'k*', 'LineWidth', 3)
                hold on
                grid on
                plot(t_log, data_fit_log, 'c-', 'LineWidth', 5)
                plot(t_log, linear_fit_log, 'r--', 'LineWidth', 3)
                legend({'Experimental Data' 'Non-linear squares fit, Non Linear' 'Non-linear squares fit, Linear'}, 'Location', 'eastoutside', 'Orientation', 'vertical')
                set(gca,'xscale','log')
                set(gca,'yscale','log')
                title('Non-linear Squares Fit', 'FontSize', 16)
                xlabel('Time [s]', 'FontSize', 20)
                ylabel('$\int_{0}^{t} U(t-\xi) F(\xi) d\xi$ [$m^2$]', 'Interpreter', 'latex', 'FontSize', 16)
                hold off

%                 saveas(gcf, sprintf('TempFigs/ComplianceParams/AveragedDataFit-nVoigt_%d',voigtArray(iVoigt)), 'jpg')

                % Move on to plotting the loss angle
                % First, generate a frequency array in log scale
                de0 = 2.0.*pi.*(1./dataStruct(length(Files)+kk).t_r(end));
                maxi = 2.0.*pi.*(1./mean(diff(dataStruct(length(Files)+kk).t_r)));
                omega = log_tw(de0,maxi);

                % Second, use the convolution J parameters
                % Calculate J_storage
                Jstorage = J_storage(omega,dataStruct(length(Files)+kk).convParams);

                % Calculate J_loss
                Jloss = J_loss(omega,dataStruct(length(Files)+kk).convParams);

                % Calculate the loss angle and save these results to the
                % structure variable
                theta = atan((Jloss./Jstorage).^-1).*(180/pi);

                dataStruct(length(Files)+kk).Jloss_conv = Jloss;
                dataStruct(length(Files)+kk).Jstorage_conv = Jstorage;
                dataStruct(length(Files)+kk).theta_conv = theta;

                % Lastly, the Linear J parameters
                % Calculate J_storage
                Jstorage_linear = J_storage(omega,dataStruct(length(Files)+kk).linearParams);

                % Calculate J_loss
                Jloss_linear = J_loss(omega,dataStruct(length(Files)+kk).linearParams);

                % Finally, calculate the loss angle and save these results to the
                % structure variable
                theta_linear = atan((Jloss_linear./Jstorage_linear).^-1).*(180/pi);

                dataStruct(length(Files)+kk).Jloss_linear = Jloss_linear;
                dataStruct(length(Files)+kk).Jstorage_linear = Jstorage_linear;
                dataStruct(length(Files)+kk).theta_linear = theta_linear;

                figure('position',[70 150 800 600]);    
                semilogx(omega, theta, 'c', 'LineWidth', 5)
                hold on
                grid on
                semilogx(omega, theta_linear, 'r--', 'LineWidth', 5)
                legend({'Fit Non Linear' 'Fit Linear'})
                xlabel('$\omega, \,rad/s$', 'FontSize', 16, 'Interpreter', 'Latex')
                ylabel('$\delta(\omega), \,deg$', 'FontSize', 16, 'Interpreter', 'Latex')
                xlim([10 10^4])
                hold off

%                 saveas(gcf, sprintf('TempFigs/ComplianceParams/AveragedLossAngle-nVoigt_%d-LoadLevel%d',voigtArray(iVoigt),kk), 'jpg')
            end

            %close all

    end
    
end

%% Delete Parallel Pool of MATLAB Workers
poolobj = gcp('nocreate');
delete(poolobj);
