function y = D_df(omega_m, p)
% D_DF Df noise power spectral density including interaction contribution
%
%   Y = D_DF(omega_m, p) calculates the noise power spectral density of  
%     the frequency shift (df) signal as a function of the modulation 
%     frequency omega_m with omega = omega_0 + omega_m and the cantilever 
%     eigenfrequency omega_0.
%     p is the parameter struct, see parameters_default. 
%     The formula includes following ontributions within an NC-AFM setup 
%     based on [1,2]:
%     - thermal cantilever noise (see also D_exc_th)
%     - detection system noise (see also p.SYS.Dzds)
%     - tip-sample interaction noise due to coupling the amplitude with the
%     frequency control loop (see also p.TS.alpha, p.TS.beta)
%
%   Following entries are accessed in parameter struct p:
%   p.CL.f0           : cantilever eigenfrequency (in Hz)
%   p.CL.Q            : cantilever quality factor
%   p.CL.k            : cantilever stiffness (in N/m)
%   p.CL.T            : cantilever temperature (in K)
%   p.CL.A            : cantilever oscillation amplitude zero-peak (in m)
%   p.ZFB.Szp         : Piezo sensitivity including amplifier gain (in m/V)
%   p.ZFB.Pgain       : proportional gain for the PID loop
%   p.ZFB.Igain       : integral gain for the PID loop
%   p.ZFB.Dgain       : differential gain for the PID loop
%   p.AFB.Pgain       : proportional gain for the PID loop
%   p.AFB.Igain       : integral gain for the PID loop
%   p.AFB.Dgain       : differential gain for the PID loop
%   p.AFB.filter_type : filter type of low-pass filter
%   p.AFB.filter_fc   : cutoff frequency of low-pass filter (in Hz)
%   p.PLL.Sdf         : Df scaling on PLL input (in Hz/V)
%   p.PLL.filter_type : filter type of loop filter
%   p.PLL.filter_fc   : cutoff frequency of loop filter (in Hz)
%   p.PLL.Pgain       : proportional gain for the PID loop
%   p.PLL.Igain       : integral gain for the PID loop
%   p.PLL.Dgain       : differential gain for the PID loop
%   p.TS.alpha        : parameter alpha describing the tip-sample
%                       interaction
%   p.TS.beta         : parameter beta describing the tip-sample
%                       interaction
%   p.SYS.dzds        : detection system noise amplitude spectral density
%   p.SYS.S_z_A   : sensitivity of detection system in m/V
%   p.SYS.S_z_exc : sensitivity of the drive piezo in m/V
%
%   [1] Polesel-Maris et al., Phys. Rev. B 79, 235401 (2009)
%   [2] Luebbe et al., Beil. J. Nanotech. 4, 32 (2013)
%
%   See also H_Csimple, H_zc, H_filter, D_z_A, D_exc_th, 
%            parameters_default, D_df_free
%
%   Copyright 2016 
%   Jannis Luebbe, Matthias Temmen, Philipp Rahe, Michael Reichling
%
%   This work is licensed under the Creative Commons 
%   Attribution-NonCommercial 4.0 International License, which permits 
%   non-commercial unrestricted use, distribution and reproduction 
%   provided the original work is properly cited. To view a copy of 
%   this license, visit http://creativecommons.org/licenses/by-nc/4.0/.
%
%   The software is provided "as is", without warranty of any kind, 
%   express or implied, including but not limited to the warranties of 
%   merchantability, fitness for a particular purpose and noninfringement. 
%   In no event shall the authors or copyright holders be liable for any 
%   claim, damages or other liability, whether in an action of contract, 
%   tort or otherwise, arising from, out of or in connection with the 
%   software or the use or other dealings in the software.
%

  % cantilever transfer function
  myHC      = H_Csimple(omega_m, p);
  % Z FB transfer function
  myHz      = H_zc(omega_m, p);
  % PLL loop filter
  myHfilter = H_filter(omega_m, p);
  % noise of the cantilever oscillation amplitude
  myDz_A    = D_z_A(omega_m, p);
  % noise power spectral density of the thermal cantilever excitation 
  Dexcth  = D_exc_th(p);

  y = abs(myHfilter./(1-1./p.CL.Q.*myHfilter.*myHC-p.TS.beta.*myHz.*myHfilter)).^2 ...
      .*(p.TS.alpha.^2.*myDz_A ... 
         + 2.*(omega_m./(2.*pi.*p.CL.A)).^2.*(p.SYS.dzds.^2+abs(myHC).^2.*Dexcth));
end

