function y = D_df_free(omegam, p) 
% D_DF_FREE Df noise power spectral density without interaction contribution
%
%   Y = D_DF_FREE(omega_m, p) calculates the noise power spectral density of  
%     the frequency shift (df) signal as a function of the modulation 
%     frequency omega_m with omega = omega_0 + omega_m and the cantilever 
%     eigenfrequency omega_0.
%     p is the parameter struct, see parameters_default. 
%     The formula includes following ontributions within an NC-AFM setup 
%     based on [1,2]:
%     - thermal cantilever noise (see also D_exc_th)
%     - detection system noise (see also p.SYS.Dzds)
%     and it models the closed frequency control loop.
%
%   Following entries are accessed in parameter struct p:
%   p.CL.f0           : cantilever eigenfrequency (in Hz)
%   p.CL.Q            : cantilever quality factor
%   p.CL.k            : cantilever stiffness (in N/m)
%   p.CL.T            : cantilever temperature (in K)
%   p.CL.A            : cantilever oscillation amplitude zero-peak (in m)
%   p.PLL.filter_type : filter type of loop filter
%   p.PLL.filter_fc   : cutoff frequency of loop filter (in Hz)
%   p.PLL.Pgain       : proportional gain for the PID loop
%   p.PLL.Igain       : integral gain for the PID loop
%   p.PLL.Dgain       : differential gain for the PID loop
%   p.SYS.dzds        : detection system noise amplitude spectral density
%
%   [1] Polesel-Maris et al., Phys. Rev. B 79, 235401 (2009)
%   [2] Luebbe et al., Beil. J. Nanotech. 4, 32 (2013)
%
%   See also H_Csimple, H_zc, H_filter, D_z_A, D_exc_th, 
%            parameters_default, D_df
%
%   Copyright 2016 
%   Jannis Luebbe, Matthias Temmen, Philipp Rahe, Michael Reichling
%
%   This work is licensed under the Creative Commons 
%   Attribution-NonCommercial 4.0 International License, which permits 
%   non-commercial unrestricted use, distribution and reproduction 
%   provided the original work is properly cited. To view a copy of 
%   this license, visit http://creativecommons.org/licenses/by-nc/4.0/.
%
%   The software is provided "as is", without warranty of any kind, 
%   express or implied, including but not limited to the warranties of 
%   merchantability, fitness for a particular purpose and noninfringement. 
%   In no event shall the authors or copyright holders be liable for any 
%   claim, damages or other liability, whether in an action of contract, 
%   tort or otherwise, arising from, out of or in connection with the 
%   software or the use or other dealings in the software.
%
  
  % PLL filter
  myHfilter = H_filter(omegam, p);
  
  % CL transfer function
  myHC      = H_Csimple(omegam, p);
  % Note: Could also use the full description:
  %myHC      = H_Cfull(omega,p);
  
  % noise power spectral density of the thermal cantilever excitation
  Dexcth  = D_exc_th(p);

  % closed loop for frequency control
  y = abs( myHfilter./(1 - myHC.*myHfilter./p.CL.Q) ).^2 ...
       .*2.*(omegam./(2.*pi.*p.CL.A)).^2 ...
       .*( abs(myHC).^2.*Dexcth + p.SYS.dzds.^2 );
  
end

