function y = H_zc(omega, p)
% H_ZC Transfer function for the z distance PID controller
%
%   Y = H_ZC(omega, p) is the transfer function for the PID controller 
%     within the height feedback loop defined in frequency space.
%     p is the parameter struct, see parameters_default. The transfer
%     function follows [1].
%
%   Following entries are accessed in parameter struct p:
%   p.PLL.Sdf   : Df scaling on PLL input (in Hz/V)
%   p.ZFB.Szp   : Piezo sensitivity including amplifier gain (in m/V)
%   p.ZFB.Pgain : proportional gain for the PID loop
%   p.ZFB.Igain : integral gain for the PID loop
%   p.ZFB.Dgain : differential gain for the PID loop
%
%   [1] Polesel-Maris et al., Phys. Rev. B 79, 235401 (2009)
%
%   See also H_PID, parameters_default
%
%   Copyright 2016 
%   Jannis Luebbe, Matthias Temmen, Philipp Rahe, Michael Reichling
%
%   This work is licensed under the Creative Commons 
%   Attribution-NonCommercial 4.0 International License, which permits 
%   non-commercial unrestricted use, distribution and reproduction 
%   provided the original work is properly cited. To view a copy of 
%   this license, visit http://creativecommons.org/licenses/by-nc/4.0/.
%
%   The software is provided "as is", without warranty of any kind, 
%   express or implied, including but not limited to the warranties of 
%   merchantability, fitness for a particular purpose and noninfringement. 
%   In no event shall the authors or copyright holders be liable for any 
%   claim, damages or other liability, whether in an action of contract, 
%   tort or otherwise, arising from, out of or in connection with the 
%   software or the use or other dealings in the software.
%

  % Df scaling on input (in Hz/V)
  Sdf = p.PLL.Sdf; 
  % Piezo sensitivity and high-voltage gain (in m/V)
  Szp = p.ZFB.Szp;
  
  y = Szp./Sdf.*H_PID(omega, p.ZFB.Pgain, p.ZFB.Igain, p.ZFB.Dgain);
  
end
