function y = H_filter(omega_m, p)
% H_FILTER Frequency response of the phase-locked loop 
%
%   Y = H_FILTER(omega_m, p) is the frequency response function of the 
%     closed phase control loop and calculated from the modulation 
%     frequency omega_m with omega = omega_0 + omega_m and the cantilever
%     eigenfrequency omega_0. p is the parameter struct, 
%     see parameters_default. 
%     The transfer function follows [1].
%
%   Following entries are accessed in parameter struct p:
%   p.PLL.filter_type : filter type of loop filter
%   p.PLL.filter_fc   : cutoff frequency of loop filter (in Hz)
%   p.PLL.Pgain       : proportional gain for the PID loop
%   p.PLL.Igain       : integral gain for the PID loop
%   p.PLL.Dgain       : differential gain for the PID loop
%
%   [1] Polesel-Maris et al., Phys. Rev. B 79, 235401 (2009)
%
%   See also H_PID2, parameters_default
%
%   Copyright 2016 
%   Jannis Luebbe, Matthias Temmen, Philipp Rahe, Michael Reichling
%
%   This work is licensed under the Creative Commons 
%   Attribution-NonCommercial 4.0 International License, which permits 
%   non-commercial unrestricted use, distribution and reproduction 
%   provided the original work is properly cited. To view a copy of 
%   this license, visit http://creativecommons.org/licenses/by-nc/4.0/.
%
%   The software is provided "as is", without warranty of any kind, 
%   express or implied, including but not limited to the warranties of 
%   merchantability, fitness for a particular purpose and noninfringement. 
%   In no event shall the authors or copyright holders be liable for any 
%   claim, damages or other liability, whether in an action of contract, 
%   tort or otherwise, arising from, out of or in connection with the 
%   software or the use or other dealings in the software.
%

  % Loop filter
  myHlp = feval(p.PLL.filter_type, omega_m, 2.*pi.*(p.PLL.filter_fc));
  % PID loop
  myPID = H_PID2(omega_m, p.PLL.Pgain, p.PLL.Igain, p.PLL.Dgain);
  
  % closed loop for full PLL
  y = myPID./(1i.*omega_m./myHlp + myPID);
  
end
