function p = parameters_default()
%% PARAMETERS_DEFAULT Returns default parameter set for an NC-AFM setup
%
%   All parameters are stored in a struct p, which is organised as follows:
%   p.SYS     - system parameters
%   p.CL      - cantilever properties
%   p.PLL     - phase-locked loop properties
%   p.ZFB     - height feedback properties
%   p.AFB     - amplitude feedback properties
%   p.TS      - tip-sample interaction parameters
%
%   Copyright 2016 
%   Jannis Luebbe, Matthias Temmen, Philipp Rahe, Michael Reichling
%
%   This work is licensed under the Creative Commons 
%   Attribution-NonCommercial 4.0 International License, which permits 
%   non-commercial unrestricted use, distribution and reproduction 
%   provided the original work is properly cited. To view a copy of 
%   this license, visit http://creativecommons.org/licenses/by-nc/4.0/.
%
%   The software is provided "as is", without warranty of any kind, 
%   express or implied, including but not limited to the warranties of 
%   merchantability, fitness for a particular purpose and noninfringement. 
%   In no event shall the authors or copyright holders be liable for any 
%   claim, damages or other liability, whether in an action of contract, 
%   tort or otherwise, arising from, out of or in connection with the 
%   software or the use or other dealings in the software.
%

  p = struct('SYS', struct(), ...
             'CL', struct(), ...
             'PLL', struct(), ...
             'ZFB', struct(), ...
             'AFB', struct(), ...
             'TS', struct());
  
  % detection and excitation system parameters
  p.SYS.dzds    = 81e-15;     % Detection system noise 
                              % amplitude spectral density d_ds^z in m/sqHz
  p.SYS.S_z_A   = 52.5.*1E-9; % Sensitivity of detection system in m/V
  p.SYS.S_z_exc = 4.65e-10;   % Sensitivity of the drive piezo in m/V
      % Note:
      % follows from Q = |A_0/A(f)| = |V_a * S_z_A / (V_exc * Poleselfak)|
      %          <=> S_z_exc = V_a * S_z_A / (V_exec * Q)
      % with A(f) = A_exc / sqrt( (1-f^2/f_0^2)^2 - ...)
      % V_a is the F_N signal (in V)
      % V_exc is the excitation signal
  
  % Cantilever parameters
  p.CL.f0 = 305337.6;     % resonance frequency in Hz
  p.CL.Q  = 43900;        % quality factor (unitless)
  p.CL.k  = 32.4;         % cantilever stiffness in N/m
  p.CL.A  = 12.6e-9;      % amplitude (zero-peak) in m
  p.CL.T  = 273.15+23.8;  % CL temperature in K
  
  % PLL parameters
  p.PLL.filter_fc   = 500;      % loop filter in Hz (cutoff frequency)
  p.PLL.filter_type = 'but3';   % type and order of loop filter
  p.PLL.Pgain = -2.1.*(-368.7); % P-gain of PLL loop in Hz/deg
            % Note that the factor -368.7 is RHK specific
  p.PLL.Igain = pi.*1;          % I-gain of PLL loop in 1/s
            % Note that the factor pi is RHK specific
  p.PLL.Dgain = 0;              % D-gain of PLL loop
  p.PLL.Sdf   = -30;            % PLL output scaling in Hz/V
  
  % Z feedback parameters
  p.ZFB.Szp     = 9.36e-9;    % Piezo sensitivity in m/V
  p.ZFB.Pgain   = 0.1;        % P-gain of Z FB in 
  p.ZFB.Igain   = 1;          % I-gain of PLL loop in 1/s
  p.ZFB.Dgain   = 0;          % D-gain of PLL loop
  
  % A feedback parameters
  p.AFB.filter_fc = 500;      % loop filter in Hz (cutoff frequency)
  p.AFB.filter_type = 'but3'; % type and order of loop filter
  p.AFB.Pgain = -0.460;       % P-gain of amplitude loop in V/V
            % Note that the negative sign is RHK specific.
  p.AFB.Igain = pi*1;         % I-gain of PLL loop in 1/s
            % Note that the factor of pi is RHK specific
  p.AFB.Dgain = 0;            % D-gain of PLL loop
  
  % tip-sample interaction
  p.TS.alpha = -12.3e9;        % alpha_ts in Hz/m
  p.TS.beta  = 12.3e9;        % beta_ts in Hz/m
  
  
end

