/*************************** Trimodal_AFM_With_Enhanced_Q3D.c *********************************/

// Programmed by Santiago D. Solares - Please direct any questions to ssolares@gwu.edu
// NOTE: This program was edited using Microsoft Visual Studio - margin alignment may be lost in other editors. 


/* HEADERS */

#include<stdio.h>
#include<math.h>


/* CONSTANTS */

#define 	pi	3.1415926535
#define		TipRadius 20.00						// Tip radius in nm
#define		PartitionSize 300					// Number of 'rings' (radial elements) in the radial partition
#define		NoPoints 30							// Number of points used in the construction of the spectroscopy curve
#define		vdWStrength 7.5e-11					// Strength of van der Waals interactions (nN/nm^2)


/* GLOBAL VARIABLES */

long int	counter1,counter2;					// counter variables
long int	print_counter1;						// Variable that controls the printing step
double		startprint;							// Simulation time when the printing starts
double		printstep;							// How often printing to the output file will occur
double		length;								// Total dynamics run time
double		Zeq;								// Cantilever equilibrium position
double		Zeq_max;							// Maximum cantilever equilibrium position (starting point of the spectroscopy curve simulation)
double		Zt_m1, Zt_m2, Zt_m3, Zto_m1, Zto_m2, Zto_m3;			// Position at time t and at time zero for the various eigenmodes
double		Z_1_m1, Z_1_m2, Z_1_m3, Z_2_m1, Z_2_m2, Z_2_m3;			// Position at the previous timestep for the various eigenmodes
double		Vo_m1, Vo_m2, Vo_m3, Vt_m1, Vt_m2, Vt_m3;     			// Initial velocity and velocity at time t for the various eigenmodes
double		at_m1, at_m2, at_m3;									// acceleration at time t for the various eigenmodes
double		time, timestep;							// Simulation time and timestep
double		time1, time2, time3;					// Times used to self-excite individual eigenmodes (currently self-excitation is not implemented)
double		mass;									// Equivalent mass 
double		Freqo_1, Freqo_2, Freqo_3;				// Resonance frequencies
double		InstFreq_1, InstFreq_2, InstFreq_3;		// Instantaneous resonance frequencies
double		Ao_1, Ao_2, Ao_3;						// Target amplitude for the various eigenmodes
double      Fo_1, Fo_2, Fo_3;						// Excitation force amplitude for the various eigenmodes
double		QF_m1, QF_m2, QF_m3, kharm_m1, kharm_m2, kharm_m3;		// Quality factors and harmonic constants for the various eigenmodes
FILE		*Out1,*Out2, *Out3;						// Pointers for the output files
FILE		*Inp;									// Pointer for the input file
char		string[100];							// String variable used for reading data from the input file
char		input_file[100];						// Input file name
char		output_file_1[100];						// Output file name for every run at every height ('forces' file)
char		output_file_2[100];						// Output file name for every run at every height ('profiles' file)
char		output_file[100];						// Output file root name
char		output_Spectroscopy[100];			// Output file containing the amplitude, phase peak force and depth reached for the full range of cantilever positions
double		period_m1;							// Oscillation period of the first mode
double		period_m2;							// Oscillation period of the second mode
double		period_m3;							// Oscillation period of the third mode
double		TipPos;								// Instantaneous tip position
double		TipV;								// Instantaneous tip velocity
double		Xb[PartitionSize], Xc[PartitionSize];			// Dashpot variables: Xb is the position of the surface skin and Xc is the position of the dashpot (damper)
double		PredictedXb;									// Variable used in relaxing the SLS
double		Partition[3][PartitionSize];					// Partition of system of SLS-rings describing the subsurface
double		k1, k2, cDiss;						// SLS constants: k1 is the residual spring, k2 is the dashpot spring, and cDiss is the dashpot constant
double		kint;								// Surface parameter used to model interaction between adjacent SLS elements
double		cohesiveness[PartitionSize];		// Surface parameter: spring constant used to model interaction between adjacent SLS elements (this is the same parameter as the above, but scaled for the various area elements)
double		ForceSLS[PartitionSize];			// Repulsive force exerted by each SLS
double		TotalForceSLS;						// Total force of all SLS elements added
double		MaxForce, PeakDepth;				// Maximum tip-sample force and lowest surface position reached during one period of the fundamental oscillation
int			NN;									// Number of periods used to calculate the phase and amplitude
double		SumA, SumB, SumC, SumD, SumE, SumF;				// Variables used for calculating the phase and amplitude
double		Output_Phase1, Output_Phase2, Output_Phase3;	// Phases of the individual eigenmodes
double		Output_Amp1, Output_Amp2, Output_Amp3;			// Amplitudes of the individual eigenmodes 
long int	counterPh1, counterPh2, counterPh3;				// Counter variables used in the calculation of the phase



/* DIMENSIONS 

Distance					nm
Time						ns
Tip vertical velocity		m/s
Tip horizontal velocity 	nm/s
Force						N

*/







// BEGINNING OF THE MAIN PROGRAM

main()
{

  // Opening input file

  sprintf(input_file,"input.txt");
  Inp = fopen(input_file,"r");

  // Reading the input parameters

  do                    // OUTPUT FILE ROOT NAME 
  {
    fscanf(Inp,"%s",string);
  } while (strncmp(string,"OutputFile",10) != 0);
  fscanf(Inp,"%s",output_file);

  do                    // FUNDAMENTAL FREQUENCY
  {
    fscanf(Inp,"%s",string);
  } while (strncmp(string,"Frequency_1",11) != 0);
  fscanf(Inp,"%lf",&Freqo_1);

  do                    // FIRST HARMONIC CONSTANT
  {
    fscanf(Inp,"%s",string);
  } while (strncmp(string,"ForceConstant_1",15) != 0);
  fscanf(Inp,"%lf",&kharm_m1);
  
  do                    // FIRST QUALITY FACTOR
  {
    fscanf(Inp,"%s",string);
  } while (strncmp(string,"QualityFactor_1",15) != 0);
  fscanf(Inp,"%lf",&QF_m1);

  do                    // SECOND QUALITY FACTOR
  {
    fscanf(Inp,"%s",string);
  } while (strncmp(string,"QualityFactor_2",15) != 0);
  fscanf(Inp,"%lf",&QF_m2);

  do                    // THIRD QUALITY FACTOR
  {
    fscanf(Inp,"%s",string);
  } while (strncmp(string,"QualityFactor_3",15) != 0);
  fscanf(Inp,"%lf",&QF_m3);

  do                    // CANTILEVER REST POSITION AT TIME ZERO
  {
    fscanf(Inp,"%s",string);
  } while (strncmp(string,"CantileverRestPosition",22) != 0);
  fscanf(Inp,"%lf",&Zeq_max);

  Zeq_max = Zeq_max*1e-9;		// Converting to meters

  do                    // FIRST AMPLITUDE TARGET
  {
    fscanf(Inp,"%s",string);
  } while (strncmp(string,"TargetAmplitude_1",17) != 0);
  fscanf(Inp,"%lf",&Ao_1);

  Ao_1 = Ao_1*1e-9;		// Converting to meters

  do                    // SECOND AMPLITUDE TARGET
  {
    fscanf(Inp,"%s",string);
  } while (strncmp(string,"TargetAmplitude_2",17) != 0);
  fscanf(Inp,"%lf",&Ao_2);

  Ao_2 = Ao_2*1e-9;		// Converting to meters

  do                    // THIRD AMPLITUDE TARGET
  {
    fscanf(Inp,"%s",string);
  } while (strncmp(string,"TargetAmplitude_3",17) != 0);
  fscanf(Inp,"%lf",&Ao_3);

  Ao_3 = Ao_3*1e-9;		// Converting to meters

  do                    // RESIDUAL SLS SPRING
  {
    fscanf(Inp,"%s",string);
  } while (strncmp(string,"k1",2) != 0);
  fscanf(Inp,"%lf",&k1);

  do                    // DASHPOT SLS SPRING
  {
    fscanf(Inp,"%s",string);
  } while (strncmp(string,"k2",2) != 0);
  fscanf(Inp,"%lf",&k2);

  do                    // DASHPOT CONSTANT
  {
    fscanf(Inp,"%s",string);
  } while (strncmp(string,"cDiss",5) != 0);
  fscanf(Inp,"%lf",&cDiss);

  do                    // SURFACE COHESIVENESS PARAMETER
  {
    fscanf(Inp,"%s",string);
  } while (strncmp(string,"kint",4) != 0);
  fscanf(Inp,"%lf",&kint);



  //Output file with the amplitude, phase, peak force and maximum indentation for the full range of cantilever positions

  sprintf(output_Spectroscopy,"%s_Spectroscopy.txt",output_file);
  Out3 = fopen(output_Spectroscopy,"w");
  fprintf(Out3,"Zcf\tForce\tDepth\tAmp1\tAmp2\tAmp3\tPhase1\tPhase2\tPhase3\n");

  // Defining the number of periods used in calculating (averaging) the phase

  NN = 25;


for (counter1 = NoPoints; counter1 > 0; counter1--)		// This routine will be repeated for each value of cantilever position used in the construction of the spectroscopy curves
{

  // Initializing phase counters

  counterPh1 = 0;
  counterPh2 = 0;
  counterPh3 = 0;

  // Resetting the maximum depth and tip-sample force 

  MaxForce = 0.0;
  PeakDepth = 0.0;

  // Defining cantilever equilibrium position

  Zeq = 1.0*counter1/NoPoints*Zeq_max;

  // Timestep

  timestep = 0.01*1e-9;

  // Calculation of second and third eigenfrequencies assuming an ideal rectangular beam

  Freqo_2 = 6.25*Freqo_1;
  Freqo_3 = 17.6*Freqo_1;  

  // Calculation of the oscillation periods 

  period_m1 = 1.0/Freqo_1;
  period_m2 = period_m1/6.25;
  period_m3 = period_m1/17.6;

  // Defining run length

  length = period_m1*2.5*QF_m1;

  // Start of printing

  startprint = length - 1.0*period_m1;

  // Printstep

  printstep = 0.001*period_m1;

  // Calculating the equivalent mass from the resonance frequency and the harmonic spring constant

  mass = kharm_m1 / pow( (2*pi*Freqo_1), 2);

  // Calculation of the dynamic force constants for the second and third eigenmodes

  kharm_m2 = kharm_m1*(Freqo_2/Freqo_1)*(Freqo_2/Freqo_1);
  kharm_m3 = kharm_m1*(Freqo_3/Freqo_1)*(Freqo_3/Freqo_1);

  // Calculation of the excitation force amplitudes for the various eigenmodes

  Fo_1 = Ao_1*kharm_m1/QF_m1;
  Fo_2 = Ao_2*kharm_m2/QF_m2;
  Fo_3 = Ao_3*kharm_m3/QF_m3;

// DYNAMICS INTEGRATION SECTION

  // Initialization of position for the various eigenmodes

  Zto_m1 = 0.0001e-9;
  Zto_m2 = 0.0001e-9;
  Zto_m3 = 0.0001e-9;

  // Initialization of velocity for the various eigenmodes

  Vo_m1 = 0.0000001e-9;  
  Vo_m2 = 0.0000001e-9;
  Vo_m2 = 0.0000001e-9;

  // Initialization of additional position variables and time for the various eigenmodes

  Z_2_m3 = Zto_m3;
  Z_2_m2 = Zto_m2;
  Z_2_m1 = Zto_m1;

  Z_1_m1 = Zto_m1 + Vo_m1*timestep;
  Z_1_m2 = Zto_m2 + Vo_m2*timestep;
  Z_1_m3 = Zto_m3 + Vo_m3*timestep;

  time = 0.0;
  time1 = 0.0;
  time2 = 0.0;
  time3 = 0.0;

  // Initialization of the excitation frequencies

  InstFreq_1 = Freqo_1;
  InstFreq_2 = Freqo_2;
  InstFreq_3 = Freqo_3;

  // Defining the name of the 'Forces' output file 

  sprintf(output_file_1,"%s_Forces_%.2f.txt",output_file,Zeq*1e9);

  // Defining the name of the 'Profiles' output file (indentation profiles)

  sprintf(output_file_2,"%s_Profiles_%.2f.txt",output_file,Zeq*1e9);

  // Opening the output files for forces and profiles

  Out1 = fopen(output_file_1,"w"); 
  Out2 = fopen(output_file_2,"w"); 

  // Printing the output file headers

  //Header for 'Forces' output file
  fprintf(Out1,"time(ns)\tZeq(nm)\tZt_m1\tZt_m2\tZt_m3\tTipPos\tXb_SpecifiedPosition\tXc_SpecifiedPosition\tTotalForceSLS\tAmp1\tAmp2\tAmp3\tPh1\tPh2\tPh3\tForce_0\tForce_3\tForce_6\tForce_9\n");
  
  //Header for 'Profiles' output file
  fprintf(Out2,"position_0\tposition_20\tposition_40\tposition_60\tposition_80\tposition_100\tposition_120\tposition_140\tposition_160\tposition_180\tposition_200\tposition_220\tposition_240\tposition_260\tposition_280\tposition_299\n");

  // Initializing the print counters

  print_counter1 = 0;

  // Initializing SLS variables

  for (counter2 = 0; counter2 < PartitionSize; counter2++)
  {
    Xb[counter2] = 0.0;		// Surface position
	Xc[counter2] = 0.0;		// Dashpot position
  }

  // Filling up the Partition[3][PartitionSize] array (Partition[0] = Rmax of interval; Partition[1] = area of interval; Partition[2] = TipHeight with respect to tip apex for interval)
  
  // The tip height with respect to the apex will follow a spherical profile up to 0.95*Rtip.  After that it will simply increase with a slope of 4 to simulate a conical tip

  for (counter2 = 0; counter2 < PartitionSize; counter2++)
  {
	Partition[0][counter2] = counter2*TipRadius*0.95/(0.5*PartitionSize);		// At interval # 0.5*PartitionSize the radius will be 0.95 of the tip radius
	Partition[1][counter2] = pi*(Partition[0][counter2]+TipRadius*0.95/(0.5*PartitionSize))*(Partition[0][counter2]+TipRadius*0.95/(0.5*PartitionSize)) - pi*(Partition[0][counter2])*(Partition[0][counter2]);
	
	// Accounting for tip geometry
	
	if (counter2 < (0.5*PartitionSize))
	  Partition[2][counter2] = TipRadius - sqrt(TipRadius*TipRadius - Partition[0][counter2]*Partition[0][counter2]);
	else
	  Partition[2][counter2] = Partition[2][counter2-1] + 4*TipRadius*0.95/(0.5*PartitionSize);

	// Scaling the cohesiveness parameter linearly with the radius for the various area elements

	cohesiveness[counter2] = kint*2*pi*((counter2+1)*TipRadius*1e-9*0.95/(PartitionSize/2.0));
  }


  do	// Dynamics integration loop
  { 

    // Incrementing the time	  
	  
    time = time + timestep;
    time1 = time1 + timestep;
    time2 = time2 + timestep;
    time3 = time3 + timestep;


    // Calculating the acceleration for the three eigenmodes using the equations of motion
    
    at_m1 = ( -kharm_m1*Zt_m1 - (mass*Freqo_1*(2*pi)*Vt_m1/QF_m1) + Fo_1*cos(InstFreq_1*2*pi*time1) + Fo_2*cos(InstFreq_2*2*pi*time2) + Fo_3*cos(InstFreq_3*2*pi*time3) + TotalForceSLS) /mass;

    at_m2 = ( -kharm_m2*Zt_m2 - (mass*Freqo_2*(2*pi)*Vt_m2/QF_m2) + Fo_1*cos(InstFreq_1*2*pi*time1) + Fo_2*cos(InstFreq_2*2*pi*time2) + Fo_3*cos(InstFreq_3*2*pi*time3) + TotalForceSLS) /mass;

    at_m3 = ( -kharm_m3*Zt_m3 - (mass*Freqo_3*(2*pi)*Vt_m3/QF_m3) + Fo_1*cos(InstFreq_1*2*pi*time1) + Fo_2*cos(InstFreq_2*2*pi*time2) + Fo_3*cos(InstFreq_3*2*pi*time3) + TotalForceSLS) /mass;

	// Resetting the SLS force

	TotalForceSLS = 0.0;

    // Calculating the velocity and position using the Verlet algorithm */
    //NOTE: the Verlet algorithm provides fourth order accuracy on the position but only second order accuracy on the velocity (other algorithms may be more advantageous to use in the presence of large dissipative forces)
    
    Zt_m1 = 2*Z_1_m1 - Z_2_m1 + at_m1*timestep*timestep;
    Zt_m2 = 2*Z_1_m2 - Z_2_m2 + at_m2*timestep*timestep;
    Zt_m3 = 2*Z_1_m3 - Z_2_m3 + at_m3*timestep*timestep;

    Vt_m1 = (Zt_m1 - Z_2_m1)/2/timestep;
    Vt_m2 = (Zt_m2 - Z_2_m2)/2/timestep;
    Vt_m3 = (Zt_m3 - Z_2_m3)/2/timestep;

    // Updating Z_2 and Z_1
    
    Z_2_m1 = Z_1_m1;
    Z_2_m2 = Z_1_m2;
    Z_2_m3 = Z_1_m3;

    Z_1_m1 = Zt_m1;
    Z_1_m2 = Zt_m2;
    Z_1_m3 = Zt_m3;

    // Calculating tip position and velocity

    TipPos = Zt_m1 + Zt_m2 + Zt_m3 + Zeq;
    TipV = Vt_m1 + Vt_m2 + Vt_m3;


	// Relaxation of the SLS

	// Relaxing the first surface element

	PredictedXb = (k2*Xc[0]+cohesiveness[0]*Xb[1])/(k1+k2+cohesiveness[0]);
	  
	if (TipPos + Partition[2][0]*1e-9 > Xb[0])
    {
      if (PredictedXb > (TipPos + Partition[2][0]*1e-9))
      {
        Xb[0] = (TipPos + Partition[2][0]*1e-9);
        ForceSLS[0] = - k1*Xb[0] - k2*(Xb[0]-Xc[0]) - cohesiveness[0]*(Xb[0]-Xb[1]) - vdWStrength;							// -vdWStrength of force is added from the vdW interaction
      }
      else
      {
		if (PredictedXb <= Xb[1])
		  Xb[0] = PredictedXb;
		else
          Xb[0] = Xb[1];

		ForceSLS[0] = -vdWStrength*0.2e-9*0.2e-9/(((TipPos + Partition[2][0]*1e-9) - Xb[0])+0.2e-9)/(((TipPos + Partition[2][0]*1e-9) - Xb[0])+0.2e-9);		// only vdw interaction (no correction needed due to cohesiveness)          
      }
      Xc[0] = Xc[0] + timestep*(1/cDiss)*k2*(Xb[0]-Xc[0]);
    }

    else
    {
      Xb[0] = (TipPos + Partition[2][0]*1e-9);
      ForceSLS[0] = -k1*Xb[0] -k2*(Xb[0]-Xc[0]) - cohesiveness[0]*(Xb[0]-Xb[1]) - vdWStrength;
      Xc[0] = Xc[0] + timestep*(1/cDiss)*k2*(Xb[0]-Xc[0]);
    }

	// Relaxing all other elements, except the last one which is not relaxed (the last element is kept fixed, and thus serves as the boundary condition for the surface profile)

	for (counter2 = 1; counter2 < (PartitionSize-1); counter2++)
	{

	  PredictedXb = (k2*Xc[counter2]+cohesiveness[counter2-1]*Xb[counter2-1]+cohesiveness[counter2]*Xb[counter2+1])/(k1+k2+cohesiveness[counter2-1]+cohesiveness[counter2]);
	  
	  if (TipPos + Partition[2][counter2]*1e-9 > Xb[counter2])		// Tip surface higher than current element position
      {
        if (PredictedXb > (TipPos + Partition[2][counter2]*1e-9))	// Predicted element position is higher than tip surface (this cannot be: the element cannot penetrate into the tip)
        {
          Xb[counter2] = (TipPos + Partition[2][counter2]*1e-9);	// Assign tip surface position to element position
          ForceSLS[counter2] = -k1*Xb[counter2] -k2*(Xb[counter2]-Xc[counter2]) -cohesiveness[counter2-1]*(Xb[counter2]-Xb[counter2-1]) -cohesiveness[counter2]*(Xb[counter2]-Xb[counter2+1]) - vdWStrength;							// -vdWStrength of force is added from the vdW interaction
        }
        else
        {
		  if (PredictedXb <= Xb[counter2+1])
		    Xb[counter2] = PredictedXb;
		  else
            Xb[counter2] = Xb[counter2+1];

          ForceSLS[counter2] = -vdWStrength*0.2e-9*0.2e-9/(((TipPos + Partition[2][counter2]*1e-9) - Xb[counter2])+0.2e-9)/(((TipPos + Partition[2][counter2]*1e-9) - Xb[counter2])+0.2e-9);		// only vdw interaction (no correction needed due to cohesiveness)          
        }
        Xc[counter2] = Xc[counter2] + timestep*(1/cDiss)*k2*(Xb[counter2]-Xc[counter2]);
      }

      else															// Tip surface is lower than current element position - element position must be depressed to be below the tip surface
      {
        Xb[counter2] = (TipPos + Partition[2][counter2]*1e-9);
        ForceSLS[counter2] = -k1*Xb[counter2] -k2*(Xb[counter2]-Xc[counter2]) -cohesiveness[counter2-1]*(Xb[counter2]-Xb[counter2-1]) -cohesiveness[counter2]*(Xb[counter2]-Xb[counter2+1]) - vdWStrength;
        Xc[counter2] = Xc[counter2] + timestep*(1/cDiss)*k2*(Xb[counter2]-Xc[counter2]);
      }
	}

	// Calculating the total force from all SLS contributions

	for (counter2 = 0; counter2 < PartitionSize; counter2++)
      TotalForceSLS = TotalForceSLS + ForceSLS[counter2]*Partition[1][counter2];


  // Calculation of phase and amplitude for the three eigenmodes (this is only done if the simulation time is greater than 0.9*startprint)

  if (time > startprint - 2*NN*period_m1)
  {

        
    SumA = SumA+cos(Freqo_1*2*pi*time)*Zt_m1*timestep;
    SumB = SumB+sin(Freqo_1*2*pi*time)*Zt_m1*timestep;
		
    SumC= SumC+cos(Freqo_2*2*pi*time)*Zt_m2*timestep;
    SumD= SumD+sin(Freqo_2*2*pi*time)*Zt_m2*timestep;
		
    SumE= SumE+cos(Freqo_3*2*pi*time)*Zt_m3*timestep;
    SumF= SumF+sin(Freqo_3*2*pi*time)*Zt_m3*timestep;


    if (time > counterPh1*NN*period_m1)
    {
      Output_Phase1 = atan(SumB/SumA)*180/pi;
      if (Output_Phase1 < 0)
        Output_Phase1 = Output_Phase1 + 180;
      Output_Amp1 = sqrt(SumA*SumA+SumB*SumB)*Freqo_1*2/NN;
		  
      counterPh1 = counterPh1 + 1;
         
      SumA = 0;
      SumB = 0;


    }

    if (time > counterPh2*7.0*NN*period_m2)		// The number of periods for the second mode is multiplied by seven
    {
      Output_Phase2 = atan(SumD/SumC)*180/pi;
      if (Output_Phase2 < 0)
        Output_Phase2 = Output_Phase2 + 180;
      Output_Amp2 = sqrt(SumC*SumC+SumD*SumD)*Freqo_2*2/NN/7.0;

      counterPh2 = counterPh2 + 1;

      SumC = 0;
      SumD = 0;

    }

    if (time > counterPh3*20.0*NN*period_m3)		// The number of periods for the third mode is multiplied by twenty
    {
      Output_Phase3 = atan(SumF/SumE)*180/pi;
      if (Output_Phase3 < 0)
        Output_Phase3 = Output_Phase3 + 180;
      Output_Amp3 = sqrt(SumE*SumE+SumF*SumF)*Freqo_3*2/NN/20.0;

      counterPh3 = counterPh3 + 1;

      SumE = 0;
      SumF = 0;

    }

  }		// End of calculation of the phase



  // Calculation of peak force and maximum indentation (depth) during one oscillation

  if ((time>=startprint) && (time <= (length)))
  {
    if (TipPos < PeakDepth)
      PeakDepth = TipPos;
    if (TotalForceSLS > MaxForce)
      MaxForce = TotalForceSLS;
  } 

  // OUTPUT FILE PRINTING FOR EVERY CANTILEVER HEIGHT ('Forces' and 'Profiles' output files)
    
    if ((time>=startprint) && (time <= (length))) 
    {       
     
      if (time > (startprint + print_counter1*printstep))
      {
        print_counter1 = print_counter1+1;
        
		// printing of force curve and observables
		fprintf(Out1,"%.12f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\n",time,Zeq*1e9,Zt_m1*1e9,Zt_m2*1e9,Zt_m3*1e9,TipPos*1e9,Xb[0]*1e9,Xc[0]*1e9,TotalForceSLS*1e9,Output_Amp1*1e9,Output_Amp2*1e9,Output_Amp3*1e9,Output_Phase1,Output_Phase2,Output_Phase3,1e9*ForceSLS[0]*Partition[1][0],1e9*ForceSLS[3]*Partition[1][3],1e9*ForceSLS[6]*Partition[1][6],1e9*ForceSLS[9]*Partition[1][9]); 
		
		// printing of surface profiles
		fprintf(Out2,"%.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\t%10.4f\n",Xb[0]*1e9,Xb[20]*1e9,Xb[40]*1e9,Xb[60]*1e9,Xb[80]*1e9,Xb[100]*1e9,Xb[120]*1e9,Xb[140]*1e9,Xb[160]*1e9,Xb[180]*1e9,Xb[200]*1e9,Xb[220]*1e9,Xb[240]*1e9,Xb[260]*1e9,Xb[280]*1e9,Xb[299]*1e9);
      }
    }

  } while (time < (length));


  // Closing output files for each cantilever position ('Forces' and 'Profiles')

  fclose(Out1);
  fclose(Out2);

  // Writing the peak force, maximum indentation (depth), apmlitudes and phases for the entire spectroscopy curve

  fprintf(Out3,"%8.4f\t%8.4f\t%8.4f\t%8.4f\t%8.4f\t%8.4f\t%8.4f\t%8.4f\t%8.4f\n",Zeq*1e9,MaxForce*1e9,PeakDepth*1e9,Output_Amp1*1e9,Output_Amp2*1e9,Output_Amp3*1e9,Output_Phase1,Output_Phase2,Output_Phase3);
  

}	// End of FOR loop (for each cantilever position)


  // Closing input file

  fclose(Inp);
  
  // Closing the 'Spectroscopy' file

  fclose(Out3);

}	// End of MAIN
