function [Ot Zt Os Os_S Ot_f] = SimSPMFeedback(time,surface,kp_n,ki_n,varargin)

% SimSPMFeedback simulates feedback including amplifier bandwidth and piezo resonance
% 
%   This function requires the MATLAB Symbolic Math Toolbox.
%
%   [Ot Zt] = SimSPMFeedback(time,surface,kp_n,ki_n);
%   Most simple form, no resonances or amplifer badwidth moddeled
%   Inputs:
%     time : 1D vector of time values for funal numerical evaluation
%     surface : Four surfaces possible specifide by following strings
%        'Step' or 'a' : a unit step (Z(t) = 1)
%        'StepRamp' or 'b' : a unit with added ramp (Z(t) = 1 + t/10)
%        'Ramp' or 'c' : a ramp (Z(t) = t/10)
%        'Topog' or 'c' : smooth topographical feature (Z(t) = t/10)
%     kp_n : Numeric value for proportinal gain
%     ki_n : Numeric value for integral gain
%   Outputs:
%     Ot : output of PI controller evaluated at times given by vactor 'time'
%     Zt : surface value evaluated at times given by vactor 'time'
%
%   [Ot Zt] = SimSPMFeedback(time,surface,kp_n,ki_n,'Property',val);
%   Additional input pairs :
%     'CutOff' : val should be the angular frequency of amplifer 3dB point.
%        If this option is used the bandwidth of the HV piezo amplifer is
%        modeled
%     'Piezo' : val should be a vector of 2 values. First is the angular
%        eigenfrequncy of the principle eqigenmode of the piezo, or other
%        mechanical component coupled to the feedback junction (i.e. an AFM
%        sensor). Second is the quality factor of this resonance. If this
%        option is used the mechanical resonance will be modeled
%   
%   [Ot Zt Os Os_S Ot_f] = SimSPMFeedback(...)
%   Additional outputs:
%     Os : Symbolic s-space outout of the system before value substitution
%     Os_S : Symbolic s-space outout of the system after value substitution
%     Ot_f : Symbolic time domain output of the system after value substitution
%
%
%   Example of use:
%      t = linspace(0,20,1000);
%      [Ot Zt] = SimSPMFeedback(t,'a',1,1,'CutOff',6,'Piezo',[5,5]);
%      figure;
%      hold all
%      plot(t,Zt,'k','LineWidth',2);
%      plot(t,Ot);

% Released under FreeBSD licence:
%
% Copyright (c) 2013, Julian Stirling
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
%
% 1. Redistributions of source code must retain the above copyright notice, this
%    list of conditions and the following disclaimer.
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
% ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
% WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
% DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
% ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
% (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
% LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
% ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
% (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
% SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
%
% The views and conclusions contained in the software and documentation are those
% of the authors and should not be interpreted as representing official policies,
% either expressed or implied, of the FreeBSD Project.


%parsing input arguments
[cutoff piezo om_p Q_p om_c] = parseinputs(varargin{:});


%set symbolic variables
s = sym('s');
t = sym('t');
Kp=sym('Kp');
Ki=sym('Ki');
omega =sym('omega');
omega_c =sym('omega_c');
Q=sym('Q');



%set transfer functions
G = Kp/s+Ki/s^2; %PID

H=1/(1+s/(omega*Q)+s^2/(omega^2)); %piezo response
Ha = 1/(s/omega_c+1); %amplifier response

%amplifier response is 1 if cutoff is not modeled
if ~cutoff
    Ha=1;
end

%Piezp response is 1 if piezo resonance is not modeled
if ~piezo
    H=1;
end

% Pick surface
switch surface
    case {'Step','a'}
        Zt_f = t/t; %t/t =1, this is needed for surface variable to be symbolic
    case {'StepRamp','b'}
        Zt_f = 1 + t/10;
    case {'Ramp','c'}
        Zt_f =  t/10;
    case {'Topog','d'}
        Zt_f = (sin(t).*exp(-t));
    otherwise
        error('Unknown surface')
end


%Laplace transfrom surface into S-space
Z = laplace(Zt_f);

%calculate output in S-space
Os =(G*Z)/(1+Ha*H*G);



% Often Kp-Ki apears on the deonominator of a fraction with a numerator
% which also evalueates to zero. This can generate not-a-number errors. We
% avoud this by applyign a negligable change to Kp to get the limiting
% behaviour.
if kp_n==ki_n
    kp_n=kp_n*1.0000000000001;
end


Os_S = subs(Os,{Kp,Ki,omega,Q,omega_c}, ...
    {kp_n,ki_n,om_p,Q_p,om_c});

Os_S = collect(Os_S,s);

%Apply inverse laplace
Ot_f = ilaplace(Os_S,s,t);

%numerically solve any roots
Ot_f = feval(symengine, 'float', Ot_f);

%change t to a numerical variable
t=time;

%Evaluate surface
Zt = eval(Zt_f );
if length(Zt)==1 %for inputs such as steps eval retuns one value as Zt is cosntant
    Zt = ones(size(t))*Zt; %multiply by vector to get corect length
end

%evaluate surface, "real" is used as evaluated roots may result in
%negligible imaginary components due to rounding errors
Ot = real(eval(Ot_f ));

end %end of main function

%Parsing th inputs, these should be in pairs
function [cutoff piezo om_p Q_p om_c] = parseinputs(varargin)

%default switches off
cutoff = false;
piezo = false;

%returning 1 for unused quantities.
Q_p=1;
om_p=1;
om_c = 1;


switch length(varargin)
    case 0
        pairs = 0;
    case {2,4} %only a maximum of two pairs understood
        pairs = length(varargin)/2; %number of pairs
        
    otherwise
        error('Number of input arguments not understood. Total number should be 4, 6, or 8');
end

%this is abit overkill for the maximum 2 pairs but it should be easy to
%extend
for n=1:pairs
    %check pair name is a string
    if ~ischar(varargin{2*n-1})
        error('The input argument number %d must be a string!',2*n+3)
    end
    
    %check for duplicates
    if n>1
        for m=1:pairs-1
            if strcmp(varargin{2*n-1},varargin{2*m-1})
                error('Cannot specify %s twice',varargin{2*n-1})
            end
        end
    end
    
    %parse pair
    switch varargin{2*n-1}
        case 'CutOff'
            cutoff = true;
            if ~isnumeric(varargin{2*n})
                error('CutOff (angular) frequency must be a numeric value')
            end
            if length(varargin{2*n})~=1
                error('Only one CutOff (angular) frequency can be supplied')
            end
            
            om_c = varargin{2*n};
        case 'Piezo'
            piezo = true;
            vars = varargin{2*n};
            if ~isnumeric(vars)
                error('Piezo properties must be a numeric value')
            end
            if length(vars)~=2
                error(['''Piezo'' requires a numeric vector of length 2:',...
                    '[om_p,Q_p], where om_p is the resonant angular frequncy',...
                    'and Q_p is the quality factor.'])
            end
            
            om_p=vars(1);
            Q_p=vars(2);
        otherwise
            error('Option name ''%s'' not recognised. Use ''CutOff'' or ''Piezo''',varargin{2*n-1})
            
    end
end
end